/*******************************************************************************
 * Copyright (c) 2007, 2019 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *     Semantum Oy - Reorganization
 *******************************************************************************/
package org.simantics.issues.common;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.OperationCanceledException;
import org.eclipse.core.runtime.SubMonitor;
import org.simantics.Simantics;
import org.simantics.db.Issue;
import org.simantics.db.ReadGraph;
import org.simantics.db.RequestProcessor;
import org.simantics.db.Resource;
import org.simantics.db.Session;
import org.simantics.db.common.request.Queries;
import org.simantics.db.common.request.ResourceRead;
import org.simantics.db.common.utils.ListUtils;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.request.PossibleActiveModel;
import org.simantics.db.layer0.util.ModelTransferableGraphSourceRequest;
import org.simantics.db.layer0.util.SessionGarbageCollection;
import org.simantics.issues.ontology.IssueResource;
import org.simantics.issues.preferences.IssuePreferenceUtil;

/**
 * @author Tuukka Lehtonen
 */
public class RunActiveValidations {

    public static void run(IProgressMonitor monitor) throws DatabaseException {
        Resource model = Simantics.getSession().syncRequest(new PossibleActiveModel(Simantics.getProjectResource()));
        if (model != null)
            run(monitor, model);
    }

    public static void run(IProgressMonitor monitor, Resource model) throws DatabaseException {

        final Session session = Simantics.getSession();

        // 1. query for which composites to run the validation
        final Collection<BatchIssueSource> validations = new ArrayList<>();
        final BatchIssueValidationContext context = new BatchIssueValidationContext();

        try {
            toBatchIssueSources(session,
                    session.syncRequest(new SelectedModelBatchIssueSources(model)),
                    validations);

            SubMonitor.convert(monitor, Messages.RunActiveValidations_MonitorPreparingResourcesForValidation, 100);
            context.contexts = Collections.singletonList(model);
            context.domain = ModelTransferableGraphSourceRequest.getDomainOnly(session, monitor, model);

            if (monitor.isCanceled())
                throw new OperationCanceledException();

        } finally {
            monitor.done();
        }

        if (!validations.isEmpty() && !context.contexts.isEmpty())
            run(monitor, validations, context);

    }

    static Collection<BatchIssueSource> toBatchIssueSources(RequestProcessor processor, Collection<Resource> sources, Collection<BatchIssueSource> result) throws DatabaseException {
        for (Resource source : sources) {
            BatchIssueSource bis = processor.syncRequest(Queries.adapt(source, BatchIssueSource.class, true));
            if (bis != null)
                result.add(bis);
        }
        return result;
    }

    public static void run(IProgressMonitor monitor, final Collection<BatchIssueSource> validations, final BatchIssueValidationContext context) throws DatabaseException {
        SubMonitor progress = SubMonitor.convert(monitor, Messages.RunActiveValidations_ValidateModel, 100);
        int maxWrittenIssues = IssuePreferenceUtil.getPreferences().maxBatchIssuesToWrite;
        int writtenIssues = 0;
        for (BatchIssueSource source : validations) {
            Map<Resource, Set<Issue>> results = BatchValidations.validate(progress.newChild(90, SubMonitor.SUPPRESS_NONE), source, context);
            if (progress.isCanceled())
                throw new OperationCanceledException();

            Collection<Resource> removed = Simantics.getSession().syncRequest(new ResourceRead<Collection<Resource>>(source.getResource()) {
                @Override
                public Collection<Resource> perform(ReadGraph graph) throws DatabaseException {
                    IssueResource ISSUE = IssueResource.getInstance(graph);
                    ArrayList<Resource> result = new ArrayList<>();
                    for (Resource issue : graph.syncRequest(new ManagedIssues(resource))) {
                        Resource list = graph.getSingleObject(issue, ISSUE.Issue_HasContexts);
                        List<Resource> l = ListUtils.toList(graph, list);
                        if (l.size() > 0) {
                            Resource mainContext = l.get(0);
                            if (!BatchValidations.isLinkedToOtherThan(graph, mainContext, issue))
                                result.add(mainContext);
                        }
                    }
                    return result;
                }
            });

            for (Resource r : removed) {
                results.put(r, Collections.<Issue>emptySet());
            }
            if (progress.isCanceled())
                throw new OperationCanceledException();

            int wroteIssues = BatchValidations.store(progress.newChild(10, SubMonitor.SUPPRESS_NONE), source.getResource(), results, Math.max(0, maxWrittenIssues - writtenIssues));
            writtenIssues += wroteIssues;

            // Try to keep resource consumption down.
            SessionGarbageCollection.gc(null, Simantics.getSession(), true, null);

        }

    }

}
