/*******************************************************************************
 * Copyright (c) 2007, 2019 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *     Semantum Oy - Reorganization
 *******************************************************************************/
package org.simantics.issues.common;

import java.util.Collection;
import java.util.Collections;
import java.util.Set;

import org.simantics.Simantics;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.common.request.BinaryRead;
import org.simantics.db.common.request.ObjectsWithType;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.adapter.Instances;
import org.simantics.db.layer0.variable.Variable;
import org.simantics.db.layer0.variable.Variables;
import org.simantics.issues.common.preferences.IssuePrefs;
import org.simantics.issues.ontology.IssueResource;
import org.simantics.layer0.Layer0;
import org.simantics.operation.Layer0X;
import org.simantics.simulation.ontology.SimulationResource;

import gnu.trove.map.hash.TObjectByteHashMap;
import gnu.trove.set.hash.THashSet;

/**
 * @author Tuukka Lehtonen
 */
public class ModelVisibleIssues extends BinaryRead<Resource, Boolean, Set<Variable>> {

    public ModelVisibleIssues(Resource model) {
        super(model, Boolean.FALSE);
    }

    public ModelVisibleIssues(Resource model, boolean onlyUnresolved) {
        super(model, onlyUnresolved);
    }

    @Override
    public Set<Variable> perform(ReadGraph graph) throws DatabaseException {

        //System.err.println("ModelVisibleIssues for " + graph.getURI(parameter));

        Layer0 L0 = Layer0.getInstance(graph);
        IssueResource ISSUE = IssueResource.getInstance(graph);

        Resource project = Simantics.getProjectResource();
        boolean showHidden = false;
        boolean showNormal = true;
        boolean showUser = true;
        if (project != null) {
            showHidden = IssuePrefs.showHiddenIssues(graph, project);
            showNormal = IssuePrefs.showNormalIssues(graph, project);
            showUser = IssuePrefs.showUserIssues(graph, project);
        }

        Instances issueIndex = graph.getPossibleAdapter(ISSUE.Issue, Instances.class);

        // Cache for source activeness.
        // 0 == not in cache, 1 == false, 2 == true
        TObjectByteHashMap<Resource> sourceActivenessCache = new TObjectByteHashMap<Resource>();

        Set<Variable> result = new THashSet<>(1013);

        Collection<Resource> modelIssues = graph.syncRequest(new ObjectsWithType(parameter, L0.ConsistsOf, ISSUE.Issue));
        Collection<Resource> indexedIssues = issueIndex != null ? issueIndex.find(graph, parameter) : Collections.<Resource>emptyList();
        Collection<Resource> issues = !indexedIssues.isEmpty() ? new THashSet<Resource>(modelIssues.size() + indexedIssues.size()) : modelIssues;
        if (!indexedIssues.isEmpty()) {
            issues.addAll(modelIssues);
            issues.addAll(indexedIssues);
        }

        for (Resource issue : issues) {
            // Filter out unwanted material
            boolean resolved = graph.hasStatement(issue, ISSUE.Resolved);
            if (parameter2 && resolved)
                continue;
            boolean hidden = graph.hasStatement(issue, ISSUE.Hidden);
            boolean user = graph.hasStatement(issue, ISSUE.UserIssue);
            boolean normal = !hidden && !user;
            if (!showHidden && hidden)
                continue;
            if (!showUser && user)
                continue;
            if (!showNormal && normal)
                continue;

            Resource source = graph.getPossibleObject(issue, ISSUE.IssueSource_Manages_Inverse);
            if (source != null) {
                byte cache = sourceActivenessCache.get(source);
                boolean active = cache == 2 ? true : false;
                if (cache == 0) {
                    active = Boolean.TRUE.equals(graph.getPossibleRelatedValue(source, ISSUE.IssueSource_active));
                    sourceActivenessCache.put(source, active ? (byte) 2 : (byte) 1);
                }
                if (!active)
                    continue;
            }

            Variable var = Variables.getPossibleVariable(graph, issue);
            if (var != null)
                result.add(var);
        }

        // System.out.println("ModelVisibleIssues returned " + result.size());
        return result;
    }

}
