/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.issues.common;

import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import org.simantics.databoard.Bindings;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.ResourceSet;
import org.simantics.db.VirtualGraph;
import org.simantics.db.WriteGraph;
import org.simantics.db.common.procedure.adapter.TransientCacheAsyncListener;
import org.simantics.db.common.procedure.adapter.TransientCacheListener;
import org.simantics.db.common.request.BinaryRead;
import org.simantics.db.common.request.DependentInstances3;
import org.simantics.db.common.request.ResourceSetGraph;
import org.simantics.db.common.request.WriteRequest;
import org.simantics.db.common.utils.NameUtils;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.changeset.MetadataUtils;
import org.simantics.db.layer0.genericrelation.DependencyChanges;
import org.simantics.db.layer0.genericrelation.DependencyChanges.Change;
import org.simantics.db.layer0.genericrelation.DependencyChanges.ComponentAddition;
import org.simantics.db.layer0.genericrelation.DependencyChanges.ComponentModification;
import org.simantics.db.layer0.genericrelation.DependencyChanges.ComponentRemoval;
import org.simantics.db.service.CollectionSupport;
import org.simantics.db.service.ManagementSupport;
import org.simantics.db.service.VirtualGraphSupport;
import org.simantics.issues.ontology.IssueResource;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class IssueSourceUtils {

    private static final Logger LOGGER = LoggerFactory.getLogger(IssueSourceUtils.class);
	public static final boolean DEBUG = false;
	public static final boolean DEBUG_CHANGE_ENTRIES = false;
	public static final boolean DEBUG_RESOURCES = false;

	public static void update(WriteGraph graph, final Resource source) throws DatabaseException {

		IssueResource ISSUE = IssueResource.getInstance(graph);
		
		Long revisionId = graph.getPossibleRelatedValue(source, ISSUE.ContinuousIssueSource_lastUpdateRevision, Bindings.LONG);
		if(revisionId == null) revisionId = 0L;
		
		ManagementSupport support = graph.getService(ManagementSupport.class);
		final long headRevision = support.getHeadRevisionId()+1;
		
		if(revisionId < headRevision) {
			VirtualGraphSupport vgs = graph.getService(VirtualGraphSupport.class);
	        VirtualGraph vg = vgs.getWorkspacePersistent(IssueConstants.ISSUE_VG);
	        graph.syncRequest(new WriteRequest(vg) {

				@Override
				public void perform(WriteGraph graph) throws DatabaseException {
					IssueResource ISSUE = IssueResource.getInstance(graph);
					graph.claimLiteral(source, ISSUE.ContinuousIssueSource_lastUpdateRevision, headRevision, Bindings.LONG);
					if(DEBUG)
						LOGGER.info("-update[" + NameUtils.getSafeName(graph, source) + "][" + headRevision + "]");
				}

			});
		}
		
	}
	
	public static boolean hasChanges(ReadGraph graph, DependencyChanges event, Resource root, Resource from) throws DatabaseException {
		Change[] changes = event.get(root);
		if(changes == null) return false;
		for(Change c : changes) {
			if(c instanceof ComponentAddition) {
				return true;
			} else if(c instanceof ComponentRemoval) {
				return true;
			} else if(c instanceof ComponentModification) {
				return true;
			}
		}
		return false;
	}

    public static class Test3 extends BinaryRead<Resource, ResourceSet, Collection<Resource>> {

        public Test3(Resource component, ResourceSet types) {
            super(component, types);
        }

        @Override
        public Collection<Resource> perform(ReadGraph graph) throws DatabaseException {
            ResourceSetGraph rsg = graph.syncRequest(new DependentInstances3(parameter), TransientCacheListener.<ResourceSetGraph>instance());
        	return rsg.resolve(graph, parameter2);
        }

    };

	private static void test(ReadGraph graph, Resource component, ResourceSet types, Collection<Resource> result) throws DatabaseException {
		Collection<Resource> data = graph.syncRequest(new Test3(component, types), TransientCacheAsyncListener.<Collection<Resource>>instance());
		if(DEBUG_RESOURCES) {
			for(Resource r : data) {
				LOGGER.info("IssueSourceUtils.test " + " " + NameUtils.getSafeName(graph, component) + " " + NameUtils.getSafeName(graph, types) + " " + NameUtils.getSafeName(graph, r));
			}
		}
		result.addAll(data);
	}

	public static Set<Resource> getChangedDependencies(ReadGraph graph, Resource model, Resource from, Set<Resource> types, DependencyChanges event) throws DatabaseException {

		Change[] changes = event.get(model);
		if(changes == null) return Collections.emptySet();

		CollectionSupport cs = graph.getService(CollectionSupport.class);
		ResourceSet typeSet = cs.getResourceSet(graph, types);
		
		Set<Resource> result = new HashSet<Resource>();
		for(Change c : changes) {
			processChange(graph, c, typeSet, result);
		}

		return result;
		
	}

	public static void processChange(ReadGraph graph, Change c, ResourceSet types, Set<Resource> result) throws DatabaseException {
		if(DEBUG_CHANGE_ENTRIES)
			LOGGER.info("-processChange: " + c.toString(graph));
		if(c instanceof ComponentAddition) {
			ComponentAddition addition = (ComponentAddition)c;
			test(graph, addition.component, types, result);
		} else if(c instanceof ComponentRemoval) {
			ComponentRemoval removal = (ComponentRemoval)c;
			result.add(removal.component);
		} else if(c instanceof ComponentModification) {
			ComponentModification modification = (ComponentModification)c;
			test(graph, modification.component, types, result);
		}
		
	}

	public static Set<Resource> getChangedDependencies(ReadGraph graph, Resource source, Resource model, Resource from, Set<Resource> types) throws DatabaseException {

		IssueResource ISSUE = IssueResource.getInstance(graph);
		
		Long revisionId = graph.getPossibleRelatedValue(source, ISSUE.ContinuousIssueSource_lastUpdateRevision, Bindings.LONG);
		if(revisionId == null) revisionId = 0L;

		if(DEBUG)
			LOGGER.info("getChangedDependencies[" + NameUtils.getSafeName(graph, source) + "][" + revisionId + "]");
		
		Map<Resource, Collection<Change>> index = MetadataUtils.getDependencyChangesFrom(graph, revisionId+1);

		CollectionSupport cs = graph.getService(CollectionSupport.class);
		ResourceSet typeSet = cs.getResourceSet(graph, types);

		Collection<Change> modelChanges = index.get(model);
		if(modelChanges == null) return Collections.emptySet();
		
		Set<Resource> result = new HashSet<Resource>();
		for(Change c : modelChanges) {
			processChange(graph, c, typeSet, result);
		}

		return result;
		
	}
	
}
