package org.simantics.issues.common;

import java.util.Objects;

import org.simantics.databoard.util.ObjectUtils;
import org.simantics.db.Resource;
import org.simantics.db.VirtualGraph;
import org.simantics.db.WriteGraph;
import org.simantics.db.common.CommentMetadata;
import org.simantics.db.common.request.WriteRequest;
import org.simantics.db.common.utils.NameUtils;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.exception.RuntimeDatabaseException;
import org.simantics.db.service.VirtualGraphSupport;
import org.simantics.issues.ontology.IssueResource;
import org.simantics.scl.runtime.SCLContext;
import org.simantics.scl.runtime.function.FunctionImpl1;

/**
 * @author Tuukka Lehtonen
 * @since 1.31.0
 */
class HideFunction extends FunctionImpl1<Boolean, Boolean> {

    private Resource[] issues;

    public HideFunction(Resource... issues) {
        this.issues = issues;
    }

    @Override
    public Boolean apply(Boolean hide) {
        try {
            WriteGraph graph = Objects.requireNonNull((WriteGraph) SCLContext.getCurrent().get("graph"));
            IssueResource ISSUE = IssueResource.getInstance(graph);
            boolean changed = false;
            for (Resource issue : issues)
                changed |= tag(graph, issue, ISSUE.Hidden, hide);
            return changed;
        } catch (DatabaseException e) {
            throw new RuntimeDatabaseException(e);
        }
    }

    private static boolean tag(WriteGraph graph, Resource r, Resource tagRel, boolean tag) throws DatabaseException {
        boolean isTagged = graph.hasStatement(r, tagRel);
        if (tag && !isTagged) {
            // If r is virtual, we perform tagging in the same vg
            VirtualGraphSupport vgs = graph.getService(VirtualGraphSupport.class);
            VirtualGraph vg = vgs.getGraph(graph, r);
            if (vg != null) {
                graph.sync(new WriteRequest(vg) {
                    @Override
                    public void perform(WriteGraph graph) throws DatabaseException {
                        graph.claim(r, tagRel, tagRel, r);
                        CommentMetadata cm = graph.getMetadata(CommentMetadata.class);
                        graph.addMetadata(cm.add(ObjectUtils.toString("Marking " + NameUtils.getSafeLabel(graph, r) + " to " + NameUtils.getSafeName(graph, tagRel))));
                    }
                });
            } else {
                graph.claim(r, tagRel, tagRel, r);
                CommentMetadata cm = graph.getMetadata(CommentMetadata.class);
                graph.addMetadata(cm.add(ObjectUtils.toString("Marking " + NameUtils.getSafeLabel(graph, r) + " to " + NameUtils.getSafeName(graph, tagRel))));
            }
            return true;
        } else if (!tag && isTagged) {
            graph.deny(r, tagRel, tagRel, r);
            CommentMetadata cm = graph.getMetadata(CommentMetadata.class);
            graph.addMetadata(cm.add(ObjectUtils.toString("Denying " + NameUtils.getSafeLabel(graph, r) + " to " + NameUtils.getSafeName(graph, tagRel))));
            return true;
        }

        return false;
    }

}