/*******************************************************************************
 * Copyright (c) 2007, 2011 Association for Decentralized Information Management in
 * Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.history.util;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.math.BigDecimal;
import java.math.MathContext;
import java.util.ArrayList;
import java.util.List;

import org.simantics.databoard.Bindings;
import org.simantics.databoard.accessor.StreamAccessor;
import org.simantics.databoard.accessor.error.AccessorException;
import org.simantics.databoard.binding.Binding;
import org.simantics.databoard.binding.RecordBinding;
import org.simantics.databoard.binding.error.BindingException;
import org.simantics.databoard.binding.mutable.Variant;
import org.simantics.databoard.serialization.Serializer;
import org.simantics.databoard.serialization.SerializerConstructionException;
import org.simantics.databoard.type.Datatype;
import org.simantics.databoard.util.Bean;
import org.simantics.databoard.util.binary.BinaryReadable;
import org.simantics.databoard.util.binary.InputStreamReadable;
import org.simantics.history.HistoryAndCollectorItem;
import org.simantics.history.HistoryException;
import org.simantics.history.HistoryManager;

/**
 * Util for import/export.
 * 
 * @author toni.kalajainen
 */
public class HistoryExportUtil {

	Binding beanArrayBinding = Bindings.getBindingUnchecked(Bean[].class);
	Serializer beanArraySerializer = Bindings.getSerializerUnchecked( beanArrayBinding );
	
	Serializer intSerializer = Bindings.getSerializerUnchecked( Bindings.INTEGER );
	
	/**
	 * Export history to an output stream.
	 * 
	 * @param history
	 * @param timeBinding
	 * @param from
	 * @param end
	 * @param out
	 * @throws IOException
	 * @throws HistoryException
	 */
	public void exportHistory( HistoryManager history, Binding timeBinding, Object from, Object end, OutputStream out )
	throws IOException, HistoryException
	{
		try {
			Bean[] items = history.getItems();
			// Serialize items as Variants
			beanArraySerializer.serialize(out, items);
			for (Bean item : items) {
				Datatype format = (Datatype) item.getField("format");
				String id = (String) item.getField("id");
				RecordBinding sampleBinding = Bindings.getBinding( format );
				Serializer sampleSerializer = Bindings.getSerializer(sampleBinding);
				Object sample = sampleBinding.createDefault();
				StreamAccessor sa = history.openStream(id, "r");
				Stream stream = new Stream(sa, sampleBinding);
				try {
					// Start index
					int startIndex = stream.binarySearch(Bindings.DOUBLE, from);						
					if ( startIndex < -stream.count() ) {
						intSerializer.serialize(0, out);
						continue;
					}
					if ( startIndex<0 ) startIndex = -2-startIndex;
					if ( startIndex == -1 ) startIndex = 0;
						
					// End index
					int endIndex = stream.binarySearch(Bindings.DOUBLE, end);
					if ( endIndex == -1 ) {
						intSerializer.serialize(0, out);
						continue;
					}
					if ( endIndex<0 ) endIndex = -1-endIndex;
					if ( endIndex == sa.size() ) endIndex = sa.size()-1;
					if ( endIndex<startIndex ) {
						intSerializer.serialize(0, out);
						continue;
					}
						
					// Write sample count
					int count = endIndex - startIndex + 1;
					intSerializer.serialize(count, out);
						
					for (int i=0; i<count; i++) {
						// Read sample
						sa.get(i, sampleBinding, sample);
						sampleSerializer.serialize(sample, out);
					}						
				} finally {
					sa.close();
				} 
			}
		} catch (AccessorException ae) {
			throw new HistoryException( ae );
		} catch (BindingException e) {
			throw new HistoryException( e );
		} catch (SerializerConstructionException e) {
			throw new HistoryException( e );
		}
	}
	
	/**
	 * Import history from input stream 
	 * 
	 * @param history
	 * @param is
	 * @throws IOException
	 * @throws HistoryException
	 */
	public void importHistory( HistoryManager history, InputStream is )
	throws IOException, HistoryException
	{		
		try {
			List<Object> ids = new ArrayList<Object>();
			BinaryReadable in = new InputStreamReadable( is, Long.MAX_VALUE );
			Bean[] items = (Bean[]) beanArraySerializer.deserialize( in );
			for (Bean item : items) {
				StreamAccessor sa = null;
				try {
					Datatype format = (Datatype) item.getField("format");
					String id = (String) item.getField("id");
					history.create(item);
					RecordBinding sampleBinding = Bindings.getBinding( format );
					Serializer sampleSerializer = Bindings.getSerializer(sampleBinding);
					Object sample = sampleBinding.createDefault();
					sa = history.openStream(id, "rw");
					int count = (Integer) intSerializer.deserialize(in);
					ids.clear();
					for (int i=0; i<count; i++) {
						sample = sampleSerializer.deserializeToTry(in, ids, sample);
						sa.add(sampleBinding, sample);
					}
				}
				finally {
					try {
						if (sa != null)
							sa.close();
					} catch (AccessorException e) {
					}
				}
			}
		} catch (AccessorException ae) {
			throw new HistoryException( ae );
		} catch (BindingException e) {
			throw new HistoryException( e );
		} catch (SerializerConstructionException e) {
			throw new HistoryException( e );
		}
	}
	
	/**
	 * An export of the history 
	 */
	static public class HistoryExport {
		public HistoryAndCollectorItem[] subscriptionItems;
		public Variant[] items;
		/* Variant = {
		       		int count;
		       		Sample[] samples;
		       }
		 */
	}
	
	private static MathContext mc = new MathContext(15); 

	/**
	 * Performs a linear interpolation with the specified times and values.
	 * @param t1
	 * @param v1
	 * @param t2
	 * @param v2
	 * @param ts
	 * @return
	 */
	public static double biglerp(double t1, double v1, double t2, double v2, double ts) {
		assert (t1 < ts) && (ts < t2);

		// #11585: Safety for Inf/NaN numbers.
		// BigDecimal throws NumberFormatException for non-finite values.
		if (!Double.isFinite(v1) || !Double.isFinite(v2)) {
			double d1 = ts-t1;
			double d2 = t2-ts;
			return (d1 > d2) ? v1 : v2;
		}

		BigDecimal T1 = new BigDecimal(-t1, mc);
		BigDecimal T2 = new BigDecimal(t2, mc);
		BigDecimal TS = new BigDecimal(ts, mc);
		BigDecimal T = (TS.add(T1, mc)).divide(T2.add(T1, mc), mc);

		BigDecimal V1 = new BigDecimal(v1, mc);
		BigDecimal V2 = new BigDecimal(v2, mc);
		BigDecimal VS = V1.add(V2.subtract(V1, mc).multiply(T, mc), mc);

		return VS.doubleValue();
	}
	
    public static boolean contains(StreamIterator iter, double time) {
    	double start = iter.getStartTime();
    	double end = iter.getEndTime();
    	// A special case, where start == end => accept
    	if(time == start) return true;
    	else if(time < start) return false;
    	else if(time >= end) return false;
    	else return true;
    }

}
