package org.simantics.history.csv;

import java.util.ArrayList;
import java.util.List;

/**
 * Specifies the interpolation method to use when producing exported samples in
 * {@link CSVFormatter}.
 * 
 * @author Tuukka Lehtonen
 * @see CSVFormatter
 */
public enum ExportInterpolation {

	/**
	 * Directly take the last sample whose time stamp is less than or equal to
	 * the sampling time.
	 */
	PREVIOUS_SAMPLE("Previous sample", "previous"),

	/**
	 * <pre>
	 * v<sub>1</sub>               v<sub>s</sub>                   v<sub>2</sub>
	 * |----------------|--------------------|
	 * t<sub>1</sub>               t<sub>s</sub>                   t<sub>2</sub>
	 * </pre>
	 * 
	 * Here t<sub>1</sub> and t<sub>2</sub> are time stamps of those two
	 * successive samples for which t<sub>1</sub> &le; t<sub>s</sub> &le;
	 * t<sub>2</sub>.
	 * 
	 * <p>
	 * The sampling result is then:
	 * <code>v<sub>s</sub> = v<sub>1</sub>+(v<sub>2</sub>-v<sub>1</sub>)*(t<sub>s</sub>-t<sub>1</sub>)/(t<sub>2</sub>-t<sub>1</sub>).
	 */
	LINEAR_INTERPOLATION("Linear interpolation", "lerp");

	public String label;
	public String preference;
	ExportInterpolation(String label, String preference) {
		this.label = label;
		this.preference = preference;
	}
	public static List<ExportInterpolation> list() {
		return toList(LINEAR_INTERPOLATION, PREVIOUS_SAMPLE);
	}
	public static ExportInterpolation fromIndex(int index) {
		return list().get(index);
	}
	public static ExportInterpolation fromPreference(String preference) {
		for(ExportInterpolation s : list()) {
			if(s.preference.equals(preference)) return s;
		}
		return list().get(0);
	}
	public int index() {
		return list().indexOf(this);
	}
	public static ExportInterpolation getDefault() {
		return LINEAR_INTERPOLATION;
	}
	public String toPreference() {
		return preference;
	}
	private static <T> List<T> toList(T ... ts) {
		ArrayList<T> result = new ArrayList<T>();
		for(T t : ts) result.add(t);
		return result;
	}
}
