/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.g2d.participant;

import org.simantics.g2d.canvas.impl.AbstractCanvasParticipant;
import org.simantics.g2d.canvas.impl.HintReflection.HintListener;
import org.simantics.g2d.scenegraph.SceneGraphConstants;
import org.simantics.scenegraph.INode;
import org.simantics.scenegraph.g2d.G2DSceneGraph;
import org.simantics.scenegraph.g2d.nodes.NavigationNode;
import org.simantics.scenegraph.utils.Quality;
import org.simantics.utils.datastructures.hints.IHintContext.Key;
import org.simantics.utils.datastructures.hints.IHintContext.KeyOf;
import org.simantics.utils.datastructures.hints.IHintObservable;

/**
 * Rendering quality interactor controls dynamic/static rendering quality of
 * the canvas context's scene graph through {@link NavigationNode}.
 * 
 * @author Tuukka Lehtonen
 */
public class RenderingQualityInteractor extends AbstractCanvasParticipant {

    /**
     * Interpreted to be true if value is TRUE or not defined, false otherwise.
     */
    public static final Key KEY_QUALITY_INTERACTOR_ENABLED = new KeyOf(Boolean.class);

    /**
     * 
     */
    public static final Key KEY_STATIC_QUALITY = new KeyOf(Quality.class);

    public RenderingQualityInteractor() {
        super();
    }

    public Quality setStaticQuality(Quality staticQuality) {
        Quality old = getHint(KEY_STATIC_QUALITY);
        if (staticQuality != null)
            setHint(KEY_STATIC_QUALITY, staticQuality);
        else
            removeHint(KEY_STATIC_QUALITY);
        return old;
    }

    private void setStaticQuality0(Quality staticQuality) {
        if (!isEnabled())
            return;
        NavigationNode nav = getNavigationNode();
        if (nav == null)
            return;
        //System.out.println("static rendering quality: " + staticQuality);
        nav.setStaticQualityMode(staticQuality);
        nav.setDynamicQuality(staticQuality == null);
    }

    public boolean isEnabled() {
        Boolean qualityPaint = getHint(KEY_QUALITY_INTERACTOR_ENABLED);
        return !Boolean.FALSE.equals(qualityPaint);
    }

    private NavigationNode getNavigationNode() {
        G2DSceneGraph sg = getContext().getSceneGraph();
        INode node = sg.lookupNode(SceneGraphConstants.NAVIGATION_NODE_NAME);
        return node instanceof NavigationNode ? (NavigationNode) node : null;
    }

    @HintListener(Class = RenderingQualityInteractor.class, Field = "KEY_STATIC_QUALITY")
    public void hintChanged(IHintObservable sender, Key key, Object oldValue, Object newValue) {
        setStaticQuality0((Quality) newValue);
    }

    @HintListener(Class = RenderingQualityInteractor.class, Field = "KEY_STATIC_QUALITY")
    public void hintRemoved(IHintObservable sender, Key key, Object oldValue) {
        setStaticQuality0(null);
    }

}
