/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.g2d.image;

import java.awt.GraphicsConfiguration;

import org.simantics.g2d.image.Image.Feature;
import org.simantics.g2d.image.impl.AWTImage;
import org.simantics.g2d.image.impl.ImageProxy;
import org.simantics.g2d.image.impl.MipMapBufferedImage;
import org.simantics.g2d.image.impl.MipMapVRamBufferedImage;
import org.simantics.g2d.image.impl.Shadow;
import org.simantics.g2d.image.impl.VRamBufferedImage;
import org.simantics.g2d.image.impl.Shadow.ShadowParameters;

/**
 * Image utilities.
 * 
 * @See {@link ProviderUtils}
 * @See {@link Image}
 * @author Toni Kalajainen
 */
public class ImageUtils {

    /**
     * Adapt AWT {@link java.awt.image.BufferedImage} to {@link Image}
     * @param awtImage
     * @return
     */
    public static Image adaptAWT(java.awt.image.BufferedImage awtImage)
    {
        return new AWTImage(awtImage);
    }

    /**
     * Creates a memory backed buffer for an Image.
     * Use this with vector images and shadows.
     * 
     * @param source
     * @return video-ram optimized version of the symbol
     */
    public static Image createBuffer(Image source)
    {
        return new MipMapBufferedImage(source);
    }

    /**
     * Creates a proxy of the source image. Proxy has a strong reference to the
     * source but weak from source to proxy and thus a proxy prevents garbage
     * collection of the source image.
     * 
     * Example:
     *  <pre>
     *   Image i = ProviderUtils.at("https://www.simantics.org/simantics/simantics/logo.jpg");
     *   Image p1 = ImageUtils.proxy(i);
     *   Image p2 = ImageUtils.proxy(i);
     *   i = null;
     * 
     *   deliver p1 and p2 to user interface components.
     * 
     *   i stays in memory until p1 and p2 are garbage collected.
     *  </pre>
     * 
     * @param source
     * @return
     */
    public static Image createProxy(Image source)
    {
        return new ImageProxy(source);
    }

    /**
     * Creates an instance of Image that reflects to the shadow of the source Image.
     * <p>
     * Shadow is created every time the image is painted. Usage of memory buffer
     * is recommended (see createBuffer() ).
     * 
     * @param source
     * @param shadow
     * @return
     */
    public static Image createShadow(Image source, ShadowParameters shadow)
    {
        return new Shadow(source, shadow);
    }

    public static Image createShadow(Image source, ShadowParameters shadow, int width, int height)
    {
        return new Shadow(source, shadow, width, height);
    }

    /**
     * Raster symbol to video memory (lazy).
     * 
     * @param ps
     * @param gc Graphics configuration
     * @return video-ram optimized version of the symbol
     */
    public static Image createVideoMemoryBuffer(Image ps, GraphicsConfiguration gc)
    {
        if (gc==null) throw new IllegalArgumentException("null arg");
        if (ps instanceof VRamBufferedImage)
        {
            VRamBufferedImage p = (VRamBufferedImage) ps;
            if (p.getGraphicsConfiguration()!=gc) ps = p.getSource(); else return ps;
        }
        if (ps instanceof MipMapVRamBufferedImage)
        {
            MipMapVRamBufferedImage p = (MipMapVRamBufferedImage) ps;
            if (p.getGraphicsConfiguration()!=gc) ps = p.getSource(); else return ps;
        }
        if (ps.getFeatures().contains(Feature.Vector))
            return new MipMapVRamBufferedImage(ps, gc);
        else
            return new VRamBufferedImage(ps, gc);
    }




}

