/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.g2d.elementclass;

import org.simantics.g2d.element.ElementClass.Single;
import org.simantics.g2d.element.IElement;
import org.simantics.g2d.element.handler.ElementHandler;

/**
 * A tagging element handler for telling whether an element can be considered a
 * branch point of a connection.
 * 
 * @author Tuukka Lehtonen
 */
@Single
public interface BranchPoint extends ElementHandler {

    /**
     * Pass-through direction preference for branch or specifically route
     * points.
     */
    public enum Direction {
        Any, Horizontal, Vertical;

        public static Direction toDirection(boolean horizontal, boolean vertical) {
            if (horizontal && vertical)
                throw new IllegalArgumentException("branch point cannot be both horizontal and vertical");
            if (horizontal)
                return Horizontal;
            if (vertical)
                return Vertical;
            return Any;
        }

        public Direction toggleDetermined() {
            switch (this) {
                case Horizontal: return Vertical;
                case Vertical: return Horizontal;
            }
            return Any;
        }

        public Direction cycleNext() {
            Direction[] dirs = values();
            int newOrdinal = ordinal() + 1;
            return dirs[newOrdinal >= dirs.length ? 0 : newOrdinal];
        }

        public Direction cyclePrevious() {
            Direction[] dirs = values();
            int newOrdinal = ordinal() - 1;
            return dirs[newOrdinal < 0 ? dirs.length - 1 : newOrdinal];
        }
    }

    /**
     * Get the direction preference assigned for the specified branch point
     * element.
     * 
     * @param e the branch point element to get the direction preference from
     * @param defaultValue this is returned if the element has no assigned
     *        direction
     * @return direction preference
     */
    Direction getDirectionPreference(IElement e, Direction defaultValue);

    /**
     * Set the direction preference assigned for the specified branch point
     * element.
     * 
     * @param e the branch point element to get the direction preference from
     * @param value the new direction preference
     */
    void setDirectionPreference(IElement e, Direction value);

}
