/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.g2d.diagram.participant;

import java.awt.Shape;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Map.Entry;
import java.util.Set;

import org.eclipse.core.runtime.ListenerList;
import org.simantics.g2d.canvas.impl.DependencyReflection.Dependency;
import org.simantics.g2d.diagram.DiagramHints;
import org.simantics.g2d.diagram.DiagramUtils;
import org.simantics.g2d.diagram.IDiagram;
import org.simantics.g2d.diagram.handler.PickContext;
import org.simantics.g2d.diagram.handler.PickRequest;
import org.simantics.g2d.diagram.handler.PickRequest.PickPolicy;
import org.simantics.g2d.diagram.handler.TransactionContext.TransactionType;
import org.simantics.g2d.element.ElementUtils;
import org.simantics.g2d.element.IElement;
import org.simantics.scenegraph.g2d.events.EventHandlerReflection.EventHandler;
import org.simantics.scenegraph.g2d.events.command.Command;
import org.simantics.scenegraph.g2d.events.command.CommandEvent;
import org.simantics.scenegraph.g2d.events.command.Commands;
import org.simantics.utils.threads.ThreadUtils;

import gnu.trove.map.hash.TObjectIntHashMap;

/**
 * This participant handles commands that manipulate z-ordering of elements.
 * 
 * @author Toni Kalajainen
 */
public class ZOrderHandler extends AbstractDiagramParticipant {

    @Dependency Selection sel;
    @Dependency PickContext pickContext;

    private final ListenerList<ZOrderListener> zOrderListeners = new ListenerList<>(ListenerList.IDENTITY);

    public void addOrderListener(ZOrderListener listener) {
        zOrderListeners.add(listener);
    }

    public void removeOrderListener(ZOrderListener listener) {
        zOrderListeners.remove(listener);
    }

    @EventHandler(priority = 0)
    public boolean handleCommand(CommandEvent ce) {
        assertDependencies();
        Command c = ce.command;

        if (c.equals( Commands.BRING_UP )) {
            final Set<IElement> selectedElements = sel.getAllSelections();
            if (selectedElements==null || selectedElements.isEmpty()) return true;
            //Shape area = ElementUtils.getElementBoundsOnDiagram(selectedElements);
            Shape area = ElementUtils.getElementShapesOnDiagram(selectedElements);
            if (area==null) return true;
            final ArrayList<IElement> pickedElements = new ArrayList<IElement>();
            PickRequest req = new PickRequest(area).context(getContext());
            req.pickPolicy = PickPolicy.PICK_INTERSECTING_OBJECTS;
            pickContext.pick(diagram, req, pickedElements);
            DiagramUtils.inDiagramTransaction(diagram, TransactionType.WRITE, new Runnable() {
                @Override
                public void run() {
                    List<IElement> elements = diagram.getElements();

                    boolean changed = false;
                    int nextPos = elements.size()-1;
                    for (int i=pickedElements.size()-1; i>=0; i--)
                    {
                        IElement e = pickedElements.get(i);
                        int index = elements.indexOf(e);
                        if (index != -1 && selectedElements.contains(e)) {
                            changed |= diagram.moveTo(e, nextPos);
                        }
                        nextPos = index;
                    }
                    if (changed) {
                        notifyZOrderListeners(diagram);
                        setDirty();
                        scheduleSynchronizeElementOrder(diagram);
                    }
                }
            });
            return true;
        }
        if (c.equals( Commands.SEND_DOWN)) {
            final Set<IElement> selectedElements = sel.getAllSelections();
            if (selectedElements==null || selectedElements.isEmpty()) return true;
            //Shape area = ElementUtils.getElementBoundsOnDiagram(selectedElements);
            Shape area = ElementUtils.getElementShapesOnDiagram(selectedElements);
            if (area==null) return true;
            final ArrayList<IElement> pickedElements = new ArrayList<IElement>();
            PickRequest req = new PickRequest(area).context(getContext());
            req.pickPolicy = PickPolicy.PICK_INTERSECTING_OBJECTS;
            pickContext.pick(diagram, req, pickedElements);
            DiagramUtils.inDiagramTransaction(diagram, TransactionType.WRITE, new Runnable() {
                @Override
                public void run() {
                    List<IElement> elements = diagram.getElements();

                    boolean changed = false;
                    int nextPos = 0;
                    for (int i=0; i<pickedElements.size(); i++)
                    {
                        IElement e = pickedElements.get(i);
                        int index = elements.indexOf(e);
                        if (index != -1 && selectedElements.contains(e)) {
                            changed |= diagram.moveTo(e, nextPos);
                        }
                        nextPos = index;
                    }
                    if (changed) {
                        notifyZOrderListeners(diagram);
                        setDirty();
                        scheduleSynchronizeElementOrder(diagram);
                    }
                }
            });
            return true;
        }
        if (c.equals( Commands.BRING_TO_TOP) ){
            DiagramUtils.inDiagramTransaction(diagram, TransactionType.WRITE, new Runnable() {
                @Override
                public void run() {
                    boolean changed = false;
                    for (Entry<Integer, Set<IElement>> e : sel.getSelections().entrySet())
                    {
                        ArrayList<IElement> ll = new ArrayList<IElement>(e.getValue());
                        _sortByOrder(ll);
                        for (int i=ll.size()-1; i>=0; i--) {
                            if (diagram.containsElement(ll.get(i)))
                                changed |= diagram.bringToTop(ll.get(i));
                        }
                    }
                    if (changed) {
                        notifyZOrderListeners(diagram);
                        setDirty();
                        scheduleSynchronizeElementOrder(diagram);
                    }
                }
            });
            return true;
        }
        if (c.equals( Commands.SEND_TO_BOTTOM) ){
            DiagramUtils.inDiagramTransaction(diagram, TransactionType.WRITE, new Runnable() {
                @Override
                public void run() {
                    boolean changed = false;
                    for (Entry<Integer, Set<IElement>> e : sel.getSelections().entrySet())
                    {
                        ArrayList<IElement> ll = new ArrayList<IElement>(e.getValue());
                        _sortByOrder(ll);
                        for (int i=0; i<ll.size(); i++) {
                            if (diagram.containsElement(ll.get(i)))
                                changed |= diagram.sendToBottom(ll.get(i));
                        }
                    }
                    if (changed) {
                        notifyZOrderListeners(diagram);
                        setDirty();
                        scheduleSynchronizeElementOrder(diagram);
                    }
                }
            });
            return true;
        }
        return false;
    }

    void scheduleSynchronizeElementOrder(final IDiagram diagram) {
        // Make sure that a mutator exists before doing this.
        if (diagram.getHint(DiagramHints.KEY_MUTATOR) == null)
            return;

        // Perform back-end write in background thread to make the UI more
        // responsive.
        ThreadUtils.getBlockingWorkExecutor().execute(() -> {
            DiagramUtils.mutateDiagram(diagram, m -> m.synchronizeElementOrder());
                //System.out.println("z-order synchronized to back-end");
        });
    }

    void notifyZOrderListeners(IDiagram diagram) {
        for (Object l : zOrderListeners.getListeners()) {
            ((ZOrderListener) l).orderChanged(diagram);
        }
    }

    void _sortByOrder(List<IElement> list)
    {
        List<IElement> elements = diagram.getElements();
        final TObjectIntHashMap<IElement> position = new TObjectIntHashMap<IElement>();
        for (IElement e : list)
            position.put(e, elements.indexOf(e));
        Comparator<IElement> c = new Comparator<IElement>() {
            @Override
            public int compare(IElement o1, IElement o2) {
                int pos1 = position.get(o1);
                int pos2 = position.get(o2);
                return pos2-pos1;
            }
        };
        Collections.sort(list, c);
    }

}
