/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.g2d.diagram.participant;

import java.util.Map;

import org.simantics.g2d.diagram.IDiagram;
import org.simantics.g2d.diagram.IDiagram.CompositionListener;
import org.simantics.g2d.diagram.handler.CanvasListener;
import org.simantics.g2d.element.IElement;
import org.simantics.utils.datastructures.hints.HintContext;
import org.simantics.utils.datastructures.hints.IHintContext;
import org.simantics.utils.datastructures.hints.IHintContext.Key;

/**
 * DiagramParticipant sends events to CanvasMonitors (=diagram handler)
 * 
 * DiagramParticipant must be added to canvas context before diagram is set.
 * 
 * @author Toni Kalajainen
 */
public class DiagramParticipant extends AbstractDiagramParticipant {

    IHintContext ctx = new HintContext();

    /**
     * Diagram composition listener
     */
    CompositionListener listener = new CompositionListener() {
        @Override
        public void onElementAdded(IDiagram d, IElement e) {

        }
        @Override
        public void onElementRemoved(IDiagram d, IElement e) {
            // TODO Remove element variables...  somehow efficiently
        }
    };

    @Override
    protected void onDiagramSet(IDiagram newDiagram, IDiagram oldDiagram) {
        if (clazz==null) return;
        for (CanvasListener cm : clazz.getItemsByClass(CanvasListener.class))
            cm.onAddedToCanvas(newDiagram, getContext());

        if (newDiagram==null)
            for (Key key : ctx.getHints().keySet())
                ctx.removeHint(key);

        if (oldDiagram!=null) oldDiagram.removeCompositionListener(listener);
        if (newDiagram!=null) newDiagram.addCompositionListener(listener);
    }

    public void setElementHint(IElement e, Key key, Object value)
    {
        Key elementCtxKey = new ElementCtxKey(e);
        IHintContext elementCtx = ctx.getHint(elementCtxKey);
        if (elementCtx==null) {
            elementCtx = new HintContext();
            ctx.setHint(elementCtxKey, elementCtx);
        }
        elementCtx.setHint(key, value);
    }

    public <E> E getElementHint(IElement e, Key key)
    {
        Key elementCtxKey = new ElementCtxKey(e);
        IHintContext elementCtx = ctx.getHint(elementCtxKey);
        if (elementCtx==null) return null;
        return elementCtx.getHint(key);
    }

    public Map<Key, Object> getElementHints(IElement e)
    {
        Key elementCtxKey = new ElementCtxKey(e);
        IHintContext elementCtx = ctx.getHint(elementCtxKey);
        if (elementCtx==null) return null;
        return elementCtx.getHints();
    }

    public <E> E removeElementHint(IElement e, Key key)
    {
        Key elementCtxKey = new ElementCtxKey(e);
        IHintContext elementCtx = ctx.getHint(elementCtxKey);
        if (elementCtx==null) return null;
        return elementCtx.removeHint(key);
    }

    public void setDiagramHint(IDiagram d, Key key, Object value)
    {
        ctx.setHint(key, value);
    }

    public <E> E removeDiagramHint(IDiagram d, Key key)
    {
        return ctx.removeHint(key);
    }

    @SuppressWarnings("unused")
    private class DiagramKey extends Key {
        public final IDiagram d;
        private final int hash;
        public final Key key;

        public DiagramKey(IDiagram d, Key key)
        {
            super();
            assert(key!=null);
            this.key = key;
            this.d = d;
            this.hash = key.hashCode() ^ d.hashCode();
        }
        @Override
        public int hashCode() {
            return hash;
        }
        @Override
        public boolean equals(Object obj) {
            if (!(obj instanceof DiagramKey)) return false;
            DiagramKey other = (DiagramKey) obj;
            if (other.key.equals(key)) return false;
            if (other.d != d) return false;
            return true;
        }
        @Override
        public boolean isValueAccepted(Object value) {
            return key.isValueAccepted(value);
        }

    }

    private class ElementKey extends Key {
        public final IElement e;
        private final int hash;
        public final Key key;

        public ElementKey(IElement e, Key key)
        {
            super();
            assert(key!=null);
            this.key = key;
            this.e = e;
            this.hash = key.hashCode() ^ e.hashCode();
        }
        @Override
        public int hashCode() {
            return hash;
        }
        @Override
        public boolean equals(Object obj) {
            if (!(obj instanceof ElementKey)) return false;
            ElementKey other = (ElementKey) obj;
            if (other.e != e) return false;
            if (!other.key.equals(key)) return false;
            return true;
        }
        @Override
        public boolean isValueAccepted(Object value) {
            return key.isValueAccepted(value);
        }
    }

    private class ElementCtxKey extends Key {
        public final IElement e;

        public ElementCtxKey(IElement e)
        {
            super();
            this.e = e;
        }
        @Override
        public int hashCode() {
            return e.hashCode();
        }
        @Override
        public boolean equals(Object obj) {
            if (!(obj instanceof ElementCtxKey)) return false;
            ElementCtxKey other = (ElementCtxKey) obj;
            return other.e == e;
        }
        @Override
        public boolean isValueAccepted(Object value) {
            return value instanceof IHintContext;
        }
    }


}
