/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.g2d.diagram.handler;

import java.util.Collection;

import org.simantics.g2d.diagram.handler.impl.TopologyImpl;
import org.simantics.g2d.element.IElement;
import org.simantics.g2d.element.handler.EdgeVisuals.EdgeEnd;
import org.simantics.g2d.element.handler.TerminalTopology;
import org.simantics.g2d.element.handler.impl.ObjectTerminal;

/**
 * 
 * TODO Remove Connection. Replace with Individual hints.
 * 
 * @see TopologyImpl Default implementation.
 * @author Toni Kalajainen
 */
public interface Topology extends DiagramHandler {

    /**
     * @See TerminalImpl
     * @see ObjectTerminal
     */
    public interface Terminal {}

    /**
     * @param edge
     * @param end Begin or End
     * @return <code>null</code> if the specified end of the specified edge is
     *         not connected
     */
    Connection getConnection(IElement edge, EdgeEnd end);

    /**
     * Get all the connections attached to the specified terminal of the
     * specified node.
     * 
     * @param node
     * @param terminal
     * @param connections
     */
    void getConnections(IElement node, Terminal terminal, Collection<Connection> connections);

    /**
     * Connect one end of a connection to an element terminal.
     * 
     * @param edge the edge to connect to the specified node, must belong to the
     *        same diagram as the node
     * @param end
     * @param node the node to connect to the specified edge, must belong to the
     *        same diagram as the edge
     * @param terminal a terminal of node, retrieved using {@link TerminalTopology}
     */
    void connect(IElement edge, EdgeEnd end, IElement node, Terminal terminal);

    /**
     * Disconnect one end of a connection from an element terminal.
     * 
     * @param edge the edge to connect to the specified node, must belong to the
     *        same diagram as the node
     * @param end the end of the edge to disconnect
     * @param node the node to disconnect the specified edge from. Must belong to the
     *        same diagram as the edge.
     * @param terminal a terminal of node, retrieved using {@link TerminalTopology}
     */
    void disconnect(IElement edge, EdgeEnd end, IElement node, Terminal terminal);

    /**
     * A hint value class for representing a single edge end to node terminal
     * connection.
     */
    public static class Connection {
        public final Terminal terminal;
        public final IElement edge;
        public final IElement node;
        public final EdgeEnd  end;
        public Connection(IElement edge, EdgeEnd end, IElement node, Terminal terminal) {
            assert edge != null;
            assert end != null;
            assert (terminal == null && node == null) || (node != null && terminal != null);
            this.edge = edge;
            this.end = end;
            this.node = node;
            this.terminal = terminal;
        }
        @Override
        public boolean equals(Object obj) {
            if (this == obj)
                return true;
            if (!(obj instanceof Connection))
                return false;
            Connection other = (Connection) obj;
            return other.terminal == terminal &&
            other.edge == edge &&
            other.node == node &&
            other.end == end;
        }
        @Override
        public int hashCode() {
            final int prime = 31;
            int result = 1;
            result = prime * result + edge.hashCode();
            result = prime * result + end.hashCode();
            result = prime * result + ((node == null) ? 0 : node.hashCode());
            result = prime * result + ((terminal == null) ? 0 : terminal.hashCode());
            return result;
        }
        @Override
        public String toString() {
            return "Topology.Connection[edge=" + edge + ", node=" + node + ", terminal=" + terminal + ", end=" + end + "]";
        }
    }

}
