/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.g2d.chassis;

import java.awt.Component;

import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.IWorkbenchWindow;
import org.simantics.g2d.canvas.ICanvasContext;
import org.simantics.g2d.event.adapter.SWTFocusAdapter;
import org.simantics.g2d.event.adapter.SWTKeyEventAdapter;
import org.simantics.g2d.event.adapter.SWTMouseEventAdapter;
import org.simantics.utils.datastructures.hints.IHintContext;
import org.simantics.utils.datastructures.hints.IHintContext.Key;
import org.simantics.utils.datastructures.hints.IHintContext.KeyOf;
import org.simantics.utils.threads.IThreadWorkQueue;
import org.simantics.utils.ui.SWTAWTComponent;

/**
 * SWT Composite based chassis
 * 
 * User must invoke CompositeChassis.syncPopulate() after constructions.
 * 
 * @author Toni Kalajainen
 */
public class SWTChassis extends SWTAWTComponent implements ICanvasChassis {

    protected AWTChassis 							component;
    protected SWTMouseEventAdapter					mouseAdapter;
    protected SWTKeyEventAdapter					keyAdapter;
    protected SWTFocusAdapter						focusAdapter;

    /**
     * Tells the Composite where the canvas is installed.
     */
    public static final Key KEY_SWT_COMPOSITE = new KeyOf(Composite.class, "SWT_COMPOSITE");

    public static final Key KEY_SWT_DISPLAY = new KeyOf(Display.class, "SWT_DISPLAY");

    public static final Key KEY_WORKBENCHWINDOW = new KeyOf(IWorkbenchWindow.class, "WORKBENCH_WINDOW");
    public static final Key KEY_WORKBENCHPAGE = new KeyOf(IWorkbenchPage.class, "WORKBENCH_PAGE");
    public static final Key KEY_EDITORPART = new KeyOf(IEditorPart.class, "EDITORPART");

    public SWTChassis(Composite parent, int style) {
        super(parent, style);
    }

    @Override
    protected void doDispose() {
        if (component!=null)
            component.fireChassisClosed();
        super.doDispose();
    }

    @Override
    protected Component createSwingComponent() {
        component = new AWTChassis(false);
        return component;
    }

    /**
     * Run in SWT thread.
     * 
     * @param canvasContext new ICanvasContext to add or null to remove
     */
    @Override
    public void setCanvasContext(ICanvasContext canvasContext) {
        assert component != null;

        // Unhook old context
        if (component.canvasContext!=null) {
            if (component.hintCtx != null) {
                component.hintCtx.removeHint( KEY_SWT_COMPOSITE );
                component.hintCtx.removeHint( KEY_SWT_DISPLAY );
            }
            // keyAdapter, mouseAdapter and focusAdapter may be null if
            // the underlying AWTChassis has already had its CanvasContext set
            // through other means than this method.
            if (keyAdapter != null) {
                removeKeyListener(keyAdapter);
                removeMouseWheelListener(mouseAdapter);
                removeMouseMoveListener(mouseAdapter);
                removeMouseTrackListener(mouseAdapter);
                removeMouseListener(mouseAdapter);
                removeFocusListener(focusAdapter);
            }
        }
        component.setCanvasContext(canvasContext);
        // Hook new context
        if (canvasContext!=null) {
            if (component.hintCtx != null) {
                component.hintCtx.setHint( KEY_SWT_COMPOSITE, this);
                component.hintCtx.setHint( KEY_SWT_DISPLAY, this.getDisplay());
            }
            mouseAdapter = new SWTMouseEventAdapter(canvasContext, canvasContext.getEventQueue());
            keyAdapter = new SWTKeyEventAdapter(canvasContext, canvasContext.getEventQueue());
            focusAdapter = new SWTFocusAdapter(canvasContext, canvasContext.getEventQueue());
            addKeyListener(keyAdapter);
            addMouseWheelListener(mouseAdapter);
            addMouseMoveListener(mouseAdapter);
            addMouseTrackListener(mouseAdapter);
            addMouseListener(mouseAdapter);
            addFocusListener(focusAdapter);
        }
    }

    @Override
    public AWTChassis getAWTComponent() {
        return component;
    }

    @Override
    public ICanvasContext getCanvasContext() {
        return component.canvasContext;
    }

    @Override
    public void addChassisListener(IThreadWorkQueue thread, IChassisListener listener) {
        component.addChassisListener(thread, listener);
    }

    @Override
    public void removeChassisListener(IThreadWorkQueue thread, IChassisListener listener) {
        component.removeChassisListener(thread, listener);
    }

    @Override
    public void addChassisListener(IChassisListener listener) {
        component.addChassisListener(listener);
    }

    @Override
    public void removeChassisListener(IChassisListener listener) {
        component.removeChassisListener(listener);
    }

    @Override
    public IHintContext getHintContext() {
        return component.hintCtx;
    }

    @Override
    public void setBackground(Color color) {
        super.setBackground(color);
        if (component != null) {
            java.awt.Color awtColor = null;
            if (color != null)
                awtColor = new java.awt.Color(color.getRed(), color.getGreen(), color.getBlue());
            component.setBackground(awtColor);
        }
    }

}
