/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.g2d.chassis;

import java.awt.BorderLayout;
import java.awt.DisplayMode;
import java.awt.Frame;
import java.awt.GraphicsDevice;
import java.awt.GraphicsEnvironment;
import java.awt.event.KeyAdapter;
import java.awt.event.KeyEvent;

import javax.swing.JFrame;

import org.eclipse.jface.viewers.ArrayContentProvider;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.window.Window;
import org.eclipse.swt.graphics.Rectangle;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Monitor;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.dialogs.ListDialog;
import org.simantics.g2d.canvas.ICanvasContext;
import org.simantics.g2d.canvas.impl.AbstractCanvasParticipant;
import org.simantics.g2d.participant.KeyToCommand;
import org.simantics.scenegraph.g2d.events.EventHandlerReflection.EventHandler;
import org.simantics.scenegraph.g2d.events.command.Command;
import org.simantics.scenegraph.g2d.events.command.CommandEvent;
import org.simantics.scenegraph.g2d.events.command.CommandKeyBinding;
import org.simantics.scenegraph.g2d.events.command.Commands;
import org.simantics.utils.datastructures.cache.IProvider;

public class FullscreenUtils {

 
	
    /**
     * Views a canvas in full screen mode
     * @param monitor display device
     * @param ctx canvas context
     * @return
     */
	public static Frame viewFullScreen(GraphicsDevice monitor, ICanvasContext ctx)
	{
        DisplayMode dm = monitor.getDisplayMode();
        final JFrame frame = new JFrame("Fullscreen mode");
        frame.setSize(dm.getWidth(), dm.getHeight());
        frame.setUndecorated(true);
		
		// This is an empty content area in the frame
		AWTChassis chassis = new AWTChassis();		
		// There is a background painter in canvas context (==it is opaque)
		chassis.setOpaque(true);
		// Esc listener
		chassis.addKeyListener(new KeyAdapter() {
			@Override
			public void keyPressed(KeyEvent e) {
				if (e.getKeyCode() == KeyEvent.VK_ESCAPE)					
					frame.dispose();
			}});
		
		frame.getContentPane().add(chassis, BorderLayout.CENTER);
		frame.pack();		

        frame.invalidate();
        java.awt.Rectangle awtBounds         = monitor.getDefaultConfiguration().getBounds();
        frame.setBounds(awtBounds);
        frame.setAlwaysOnTop(true);
		frame.setVisible(true);		
		
		chassis.setCanvasContext(ctx);
		return frame;
	}
	
	/**
	 * Adds Alt-Enter full screen handler to a canvas context
	 * @param ctx
	 */
    public static void addFullScreenHandler(ICanvasContext ctx, final Shell shell, final IProvider<ICanvasContext> fullscreenProvider)
    {
        // Key bindings
        KeyToCommand commands = ctx.getAtMostOneItemOfClass(KeyToCommand.class);
        if (commands==null)
        	ctx.add( commands = new KeyToCommand() );
        commands.addBinding( new CommandKeyBinding(Commands.FULL_SCREEN, "View in full screen mode", KeyEvent.VK_ENTER, KeyEvent.VK_CONTROL) );
        ctx.add( new AbstractCanvasParticipant() {
        	@SuppressWarnings("unused")
            @EventHandler(priority = 0)
        	public boolean handleEvent(CommandEvent e) {
        		assertDependencies();
        		Command c = e.command;
        		// Arrow key panning
        		if (c.equals(Commands.FULL_SCREEN))
        		{
        			shell.getDisplay().asyncExec(new Runnable() {
						@Override
						public void run() {
		        			GraphicsDevice gd = getOrQueryDisplay(shell);
		        			if (gd!=null) {
		        				viewFullScreen(gd, fullscreenProvider.get());
						}}});
        			return true;
        		}
        		return false;
        	}	
        });
    }	

    /**
     * Gets a single display or dialogs
     * @param shell
     * @return null or a display
     */
    public static GraphicsDevice getOrQueryDisplay(Shell shell)
    {
    	GraphicsDevice list[] = GraphicsEnvironment.getLocalGraphicsEnvironment().getScreenDevices();
    	if (list.length==0) return null;
    	if (list.length==1) return list[0];
    	return doDisplayDialog(shell);
    }
    
    /**
     * Dialog for choosing display device
     * @return
     */
    public static GraphicsDevice doDisplayDialog(Shell shell)
    {
    	ListDialog ld = new ListDialog(shell);
    	ld.setInput(GraphicsEnvironment.getLocalGraphicsEnvironment().getScreenDevices());
    	ld.setTitle("Select display device");
    	ld.setInitialSelections(new Object[] {GraphicsEnvironment.getLocalGraphicsEnvironment().getDefaultScreenDevice()});
    	ld.setMessage("Select display device");
    	ld.setContentProvider(new ArrayContentProvider());
    	ld.setLabelProvider(new LabelProvider() {
    		@Override
    		public String getText(Object element) {
    			GraphicsDevice gd = (GraphicsDevice) element;
    			return gd.getIDstring()+" ("+gd.getDisplayMode().getWidth()+"*"+gd.getDisplayMode().getHeight()+")";
    		}    		
    	});
    	ld.setBlockOnOpen(true);
    	if (ld.open() != Window.OK) return null;
    	return (GraphicsDevice)ld.getResult()[0];
    }
	
	/**
	 * Adds esc to close full screen view
	 * @param ctx
	 */
    @SuppressWarnings("unused")
    private static void addCloseHandler(ICanvasContext ctx, final Frame frame)
    {
        // Key bindings
        KeyToCommand commands = ctx.getAtMostOneItemOfClass(KeyToCommand.class);
        if (commands==null)
        	ctx.add( commands = new KeyToCommand() );
        commands.addBinding( new CommandKeyBinding(Commands.CLOSE, "Close Canvas", KeyEvent.VK_ESCAPE) );
        ctx.add( new AbstractCanvasParticipant() {
        	@EventHandler(priority = 0)
        	public boolean handleEvent(CommandEvent e) {
        		assertDependencies();
        		frame.dispose();
        		return true;
        	}	
        });
    }	    
    
    /**
     * Get the monitor the shell is mostly in
     * @param shell
     * @return
     */
    public static Monitor getMonitor(Shell shell)
    {
        Monitor result = null;
        long largestArea = 0;
               
        Rectangle shellRectangle = shell.getBounds();
        for (Monitor monitor : Display.getCurrent().getMonitors())
        {
            Rectangle monitorBounds = monitor.getBounds();
            Rectangle intersection = monitorBounds.intersection(shellRectangle);
            long area = intersection.width * intersection.height;
            if (area>largestArea) {
                largestArea = area;
                result = monitor;
            }
        }
        return result;        
    }	
	
    /**
     * Gets corresponding awt monitor for a swt monitor 
     * @param swtMonitor
     * @return
     */
    public static GraphicsDevice getMonitorCorrespondence(Monitor swtMonitor)
    {
        Rectangle swtBounds = swtMonitor.getBounds();

        for (GraphicsDevice gd : GraphicsEnvironment.getLocalGraphicsEnvironment().getScreenDevices())
        {
            if (gd.getType() != GraphicsDevice.TYPE_RASTER_SCREEN) continue;
            java.awt.Rectangle awtBounds         = gd.getDefaultConfiguration().getBounds();
            if (awtBounds.x == swtBounds.x && awtBounds.y == swtBounds.y && awtBounds.width == swtBounds.width && awtBounds.height == swtBounds.height)
                return gd;
        }
        return null;
    }
    
}
