/*******************************************************************************
 * Copyright (c) 2023, 2024 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Semantum Oy - initial API and implementation
 *******************************************************************************/
package org.simantics.document.ui.prefs;

import java.io.IOException;

import org.eclipse.core.runtime.preferences.InstanceScope;
import org.eclipse.ui.preferences.ScopedPreferenceStore;
import org.osgi.service.prefs.Preferences;
import org.simantics.document.ui.Activator;

/**
 * @author Tuukka Lehtonen
 */
public final class DocumentsPreferences {

	/**
	 * Boolean preference, default value
	 * {@value #DEFAULT_PREFER_EXTERNAL_EDITOR_FOR_DOCUMENTS}.
	 */
	public static final String P_PREFER_EXTERNAL_EDITOR_FOR_DOCUMENTS = "documents.preferExternalEditorForEmbeddedDocuments"; //$NON-NLS-1$

	public static final String P_DOCUMENTATION_VIEW_EDITOR_VISIBLE = "documents.documentationViewEditorVisible"; //$NON-NLS-1$

	/**
	 * This corresponds to how ExternalEditorAdapter has always worked in the past.
	 */
	public static final boolean DEFAULT_PREFER_EXTERNAL_EDITOR_FOR_DOCUMENTS = false;

	public static final boolean DEFAULT_DOCUMENTATION_VIEW_EDITOR_VISIBLE = true;

	private final ScopedPreferenceStore store;
	public boolean preferExternalEditorForDocuments;
	public boolean documentationViewEditorVisible;

	public static DocumentsPreferences getFactoryDefaults() {
		return new DocumentsPreferences(
				null,
				DEFAULT_PREFER_EXTERNAL_EDITOR_FOR_DOCUMENTS,
				DEFAULT_DOCUMENTATION_VIEW_EDITOR_VISIBLE);
	}

	public DocumentsPreferences(ScopedPreferenceStore store, boolean preventSimulationWithErrors, boolean documentationViewEditorVisible) {
		this.store = store;
		this.preferExternalEditorForDocuments = preventSimulationWithErrors;
		this.documentationViewEditorVisible = documentationViewEditorVisible;
	}

	public static ScopedPreferenceStore getStore() {
		return new ScopedPreferenceStore(InstanceScope.INSTANCE, Activator.PLUGIN_ID);
	}

	public static DocumentsPreferences getPreferences() {
		return getPreferences(getStore());
	}

	public static DocumentsPreferences getPreferences(ScopedPreferenceStore store) {
		boolean peefd = store.getBoolean(P_PREFER_EXTERNAL_EDITOR_FOR_DOCUMENTS);
		boolean dvev = store.getBoolean(P_DOCUMENTATION_VIEW_EDITOR_VISIBLE);
		return new DocumentsPreferences(store, peefd, dvev);
	}

	public void storePreferences() throws IOException {
		store.setValue(P_PREFER_EXTERNAL_EDITOR_FOR_DOCUMENTS, preferExternalEditorForDocuments);
		store.setValue(P_DOCUMENTATION_VIEW_EDITOR_VISIBLE, documentationViewEditorVisible);
		if (store.needsSaving())
			store.save();
	}

	public void putPreferences(Preferences node) {
		node.putBoolean(P_PREFER_EXTERNAL_EDITOR_FOR_DOCUMENTS, preferExternalEditorForDocuments);
		node.putBoolean(P_DOCUMENTATION_VIEW_EDITOR_VISIBLE, documentationViewEditorVisible);
	}

	@Override
	public String toString() {
		return String.format("[preferExternalEditorForDocuments=%b, documentationViewEditorVisible=%b]", //$NON-NLS-1$
				preferExternalEditorForDocuments, documentationViewEditorVisible);
	}

	public DocumentsPreferences readFrom(DocumentsPreferences other) {
		this.preferExternalEditorForDocuments = other.preferExternalEditorForDocuments;
		this.documentationViewEditorVisible = other.documentationViewEditorVisible;
		return this;
	}

}
