/*******************************************************************************
 * Copyright (c) 2007, 2024 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *     Semantum Oy - GitLab #1129
 *******************************************************************************/
package org.simantics.document.ui;

import java.io.IOException;
import java.util.Arrays;

import org.eclipse.core.runtime.preferences.IEclipsePreferences;
import org.eclipse.core.runtime.preferences.IEclipsePreferences.IPreferenceChangeListener;
import org.eclipse.core.runtime.preferences.IEclipsePreferences.PreferenceChangeEvent;
import org.eclipse.core.runtime.preferences.InstanceScope;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.action.IAction;
import org.eclipse.swt.browser.Browser;
import org.eclipse.swt.custom.SashForm;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Sash;
import org.eclipse.ui.IWorkbenchPart;
import org.simantics.Simantics;
import org.simantics.db.ReadGraph;
import org.simantics.document.ui.prefs.DocumentsPreferences;
import org.simantics.ui.workbench.IPropertyPage;
import org.simantics.utils.ui.BundleUtils;
import org.simantics.utils.ui.SWTUtils;
import org.simantics.views.swt.ModelledView;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * @author Antti Villberg
 */
public class DocumentView extends ModelledView {

    private static final Logger LOGGER = LoggerFactory.getLogger(DocumentView.class);

    private boolean pinSelection = false;
    private ShowEditor showEditor;

    @Override
    protected String configurationURI() {
        return DocumentUIResource.URIs.View;
    }

    @Override
    protected IPropertyPage getPropertyPage() {
        return null;
    }

    @Override
    protected void inputChanged(IWorkbenchPart provider, Object input) {
        if (pinSelection)
            return;
        super.inputChanged(provider, input);
    }

    @Override
    public void fireInput() {
        super.fireInput();
        scheduleUIRefresh();
    }

    public void scheduleUIRefresh() {
        // This is a kind of hack to try ensure that when showEdito.runWithoutUserInteraction
        // is ran, the UI should already be updated based on the new input.
        Simantics.getSession().asyncRequest((ReadGraph g) -> {
            SWTUtils.asyncExec(body, () -> {
                if (!body.isDisposed())
                    showEditor.runWithoutUserInteraction();
            });
            return null;
        });
    }

    @Override
    public void createPartControl(Composite parent) {
        super.createPartControl(parent);

        var tbm = getViewSite().getActionBars().getToolBarManager();

        tbm.add(new PinSelection());

        showEditor = new ShowEditor(DocumentsPreferences.getPreferences().documentationViewEditorVisible);
        tbm.add(showEditor);

        documentPreferenceNode = InstanceScope.INSTANCE.getNode(Activator.PLUGIN_ID);
        documentPreferenceNode.addPreferenceChangeListener(preferenceListener);
    }

    @Override
    public void dispose() {
        if (documentPreferenceNode != null) {
            documentPreferenceNode.removePreferenceChangeListener(preferenceListener);
        }
        super.dispose();
    }

    private class PinSelection extends Action {
        public PinSelection() {
            super(Messages.DocumentView_PinSelection, IAction.AS_CHECK_BOX);
            setImageDescriptor(
                    BundleUtils.getImageDescriptorFromPlugin(
                            "org.eclipse.ui", //$NON-NLS-1$
                            "icons/full/elcl16/pin_view.png")); //$NON-NLS-1$
        }

        @Override
        public void run() {
            pinSelection = isChecked();
        }
    }

    private class ShowEditor extends Action {
        public ShowEditor(boolean initialState) {
            super("Show Documentation Editor", IAction.AS_CHECK_BOX);
            setImageDescriptor(
                    BundleUtils.getImageDescriptorFromPlugin(
                            "org.eclipse.ui", //$NON-NLS-1$
                            "icons/full/etool16/editor_area.png")); //$NON-NLS-1$
            setChecked(initialState);
        }

        @Override
        public void runWithEvent(Event event) {
            // Invoked only when user presses the button in the toolbar
            restoreMaximizedState(false);

            // Set instance preference for editor control visibility
            var prefs = DocumentsPreferences.getPreferences();
            prefs.documentationViewEditorVisible = isChecked();
            try {
                prefs.storePreferences();
            } catch (IOException e) {
                LOGGER.error("Failed to persist Documentation preferences in instance scope", e); //$NON-NLS-1$
            }
        }

        @Override
        public void run() {
            restoreMaximizedState(true);
        }

        public void runWithoutUserInteraction() {
            run();
        }

        public void simulateClick() {
            restoreMaximizedState(false);
        }

        boolean restoreMaximizedState(boolean setCheckedByPreference) {
            SashForm sf = findControl(container, SashForm.class);
            LOGGER.trace("sashform: {}", sf);
            if (sf != null) {
                Control[] children = sf.getChildren();
                LOGGER.trace("children: {}", Arrays.toString(children));
                if (children.length == 3
                        && children[0] instanceof Browser
                        && children[1] instanceof Composite
                        && children[2] instanceof Sash) {

                    var prefs = DocumentsPreferences.getPreferences();
                    if (setCheckedByPreference) {
                        LOGGER.trace("set editor visible by preference: {}", prefs);
                        setChecked(prefs.documentationViewEditorVisible);
                        sf.setMaximizedControl(prefs.documentationViewEditorVisible ? null : children[0]);
                    } else {
                        boolean editorVisible = isChecked(); 
                        LOGGER.trace("set editor visible: {}", editorVisible);
                        sf.setMaximizedControl(editorVisible ? null : children[0]);
                    }
                    return true;
                }
            }
            return false;
        }
    }

    @SuppressWarnings("unchecked")
    private static <T extends Control> T findControl(Control from, Class<T> clazz) {
        if (clazz.isInstance(from))
            return (T) from;
        if (from instanceof Composite comp) {
            for (var child : comp.getChildren()) {
                T r = findControl(child, clazz);
                if (r != null)
                    return r;
            }
        }
        return null;
    }

    private IEclipsePreferences documentPreferenceNode;

    private IPreferenceChangeListener preferenceListener = new IPreferenceChangeListener() {
        @Override
        public void preferenceChange(PreferenceChangeEvent event) {
            if (DocumentsPreferences.P_DOCUMENTATION_VIEW_EDITOR_VISIBLE.equals(event.getKey())) {
                boolean editorVisible = documentPreferenceNode.getBoolean(DocumentsPreferences.P_DOCUMENTATION_VIEW_EDITOR_VISIBLE, DocumentsPreferences.DEFAULT_DOCUMENTATION_VIEW_EDITOR_VISIBLE);
                showEditor.setChecked(editorVisible);
                showEditor.simulateClick();
            }
        }
    };

}
