package org.simantics.document.server.io;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

public class CommandContextImpl implements CommandContextMutable {

	private Map<String,List<List<Object>>> data = new HashMap<String,List<List<Object>>>();

	@Override
	public boolean containsKey(String key) {
		return data.containsKey(key);
	}
	
	@Override
	public List<List<Object>> getRows(String key) {
		List<List<Object>> rows = data.get(key);
		if (rows != null) 
			return rows;
		else
			return Collections.emptyList();
	}

	@Override
	public CommandContextMutable putRow(String key, List<Object> row) {
		List<List<Object>> rows = ensureRowsAvailable(key);
		rows.add(row);
		return this;
	}

	@Override
	public String getString(String key) {
		Object o = getValue(key);
		return (String)o;
	}
	
	@Override
	public CommandContextMutable putString(String key, String value) {
	    return putValue(key, value);
	}

	private List<List<Object>> ensureRowsAvailable(String key) {
		List<List<Object>> rows = data.get(key);
		if (rows == null) {
			rows = new ArrayList<List<Object>>();
			data.put(key, rows);
		}
		return rows;
	}

	@Override
	public Map<String, List<List<Object>>> getData() {
		return data;
	}
	
	@Override
	public List<String> getKeys() {
		return new ArrayList<String>(data.keySet());
	}
	
	@Override
	public CommandContextMutable merge(CommandContext context) {
		if (context != null) {
			Map<String,List<List<Object>>> from = context.getData(); 
			for (Map.Entry<String, List<List<Object>>> entry : from.entrySet()) {
				String key = entry.getKey();
				putValue(key, context.getValue(key));
			}
		}
		return this;
	}
	
	public static CommandContextMutable create() {
		return new CommandContextImpl();
	}
	
	@Override
	public String toString() {
		StringBuilder sb = new StringBuilder();
		sb.append("CommandContext:\n");
		for (Map.Entry<String, List<List<Object>>> entry : data.entrySet()) {
			String key = entry.getKey();
			if(key.startsWith("__")) continue;
			sb.append(key);
			sb.append(":");
			List<List<Object>> value = entry.getValue();
			if(value.size() == 1) {
				List<Object> t = (List<Object>)value.get(0);
				if(t.size() == 2) {
					sb.append(t.get(1));
				} else {
					sb.append(t);
				}
			} else {
				sb.append(value);
			}
			sb.append("\n");
		}
		return sb.toString();
	}
	
	@Override
	public int hashCode() {
		final int prime = 31;
		int result = 1;
		result = prime * result + ((data == null) ? 0 : data.hashCode());
		return result;
	}

	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (getClass() != obj.getClass())
			return false;
		CommandContextImpl other = (CommandContextImpl) obj;
		if (data == null) {
			if (other.data != null)
				return false;
		} else if (!data.equals(other.data))
			return false;
		return true;
	}

    @Override
    public CommandContextMutable putValue(String key, Object value) {
        if(data.containsKey(key)) data.remove(key);
        List<List<Object>> rows = ensureRowsAvailable(key);
        List<Object> t = new ArrayList<Object>();
        t.add(key);
        t.add(value);
        rows.add(t);
        return this;
    }
    
    @Override
    public <T> T getValue(String key) {
        List<List<Object>> rows = getRows(key);
        if((rows == null) || (rows.size() != 1)) return null;
        List<Object> t = (List<Object>)rows.get(0);
        if(t.size() != 2) return null;
        @SuppressWarnings("unchecked")
        T o = (T) t.get(1);
        return o;
    }
}