/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.diagram.synchronization.graph.layer;

import java.util.HashMap;
import java.util.Map;

import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.WriteGraph;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.exception.ManyObjectsForFunctionalRelationException;
import org.simantics.db.exception.ServiceException;
import org.simantics.diagram.stubs.DiagramResource;
import org.simantics.layer0.Layer0;

/**
 * Utilities for manipulating diagram layers in the graph.
 * 
 * @author Tuukka Lehtonen
 */
public final class GraphLayerUtil implements IGraphLayerUtil {

    @Override
    public GraphLayer createLayer(WriteGraph graph, String layerName, boolean active) throws DatabaseException {
        Layer0 L0 = Layer0.getInstance(graph);
        DiagramResource DIA = DiagramResource.getInstance(graph);
        
        Resource layer = graph.newResource();
        graph.claim(layer, L0.InstanceOf, null, DIA.Layer);

        // Assign tagging relations
        Resource visibleTag = newTag(graph, L0, DIA.IsVisible);
        Resource focusableTag = newTag(graph, L0, DIA.IsFocusable);
        graph.claim(layer, DIA.HasVisibleTag, visibleTag);
        graph.claim(layer, DIA.HasFocusableTag, focusableTag);

        // Assign shared name property for all, the layer and the tags
        Resource name = graph.newResource();
        graph.claim(name, L0.InstanceOf, null, L0.String);
        graph.claimValue(name, layerName);

        graph.claim(layer, L0.HasName, name);
        graph.claim(visibleTag, L0.HasName, name);
        graph.claim(focusableTag, L0.HasName, name);

        setLayerActive(graph, DIA, layer, active);

        Map<String, Resource> properties = new HashMap<>();
        properties.put(GraphLayer.PROP_FOCUSABLE, focusableTag);
        properties.put(GraphLayer.PROP_VISIBLE, visibleTag);
        
        return new GraphLayer(layerName, layer, properties, active);
    }

    public static Resource newTag(WriteGraph graph, Layer0 L0, Resource baseTag) throws DatabaseException {
        Resource tag = graph.newResource();
        graph.claim(tag, L0.SubrelationOf, baseTag);
        graph.claim(tag, L0.InverseOf, tag);
        return tag;
    }

    public static void setLayerActive(WriteGraph graph, DiagramResource DIA, Resource layer, boolean active) throws ManyObjectsForFunctionalRelationException, ServiceException {
        graph.claimLiteral(layer, DIA.IsActive, Boolean.valueOf(active));
    }

    @Override
    public GraphLayer loadLayer(ReadGraph graph, Resource layer) throws DatabaseException {
        Layer0 L0 = Layer0.getInstance(graph);
        DiagramResource DIA = DiagramResource.getInstance(graph);
        String name = graph.getRelatedValue(layer, L0.HasName);
        Resource visible = graph.getSingleObject(layer, DIA.HasVisibleTag);
        Resource focusable = graph.getSingleObject(layer, DIA.HasFocusableTag);
        boolean active = graph.getRelatedValue(layer, DIA.IsActive);
        Map<String, Resource> properties = new HashMap<>();
        properties.put(GraphLayer.PROP_FOCUSABLE, focusable);
        properties.put(GraphLayer.PROP_VISIBLE, visible);

        return new GraphLayer(name, layer, properties, active);
    }
    
    public static void addToVisibleLayers(WriteGraph graph, Resource element, Resource diagram) throws DatabaseException {
        DiagramResource DIA = DiagramResource.getInstance(graph);
        for (Resource layer : graph.getObjects(diagram, DIA.HasLayer)) {
            boolean active = graph.getRelatedValue(layer, DIA.IsActive);
            if (active) {
                Resource visibleTag = graph.getSingleObject(layer, DIA.HasVisibleTag);
                Resource focusableTag = graph.getSingleObject(layer, DIA.HasFocusableTag);
                graph.claim(element, visibleTag, element);
                graph.claim(element, focusableTag, element);
            }
        }
    }

}
