/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.diagram.symbollibrary;

import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;

import org.simantics.db.exception.DatabaseException;
import org.simantics.diagram.synchronization.graph.BasicResources;
import org.simantics.project.IProject;
import org.simantics.project.exception.ProjectException;
import org.simantics.project.features.AbstractProjectFeature;
import org.simantics.utils.datastructures.disposable.AbstractDisposable;
import org.simantics.utils.datastructures.disposable.IDisposable;

class SymbolManager extends AbstractDisposable implements ISymbolManager, IDisposable {

    IProject project;

    public SymbolManager(IProject project) {
        this.project = project;
        project.setHint(ISymbolManager.KEY_SYMBOL_GROUPS, Collections.emptySet());
    }

    @Override
    protected void doDispose() {
        project.setHint(KEY_SYMBOL_GROUPS, Collections.emptySet());
    }

    @Override
    public void addEntryPoints(Collection<ISymbolGroup> eps) {
        Collection<ISymbolGroup> groups = project.getHint(KEY_SYMBOL_GROUPS);
        Collection<ISymbolGroup> newGroups = new HashSet<ISymbolGroup>(groups);
        newGroups.addAll(eps);
        project.setHint(KEY_SYMBOL_GROUPS, Collections.unmodifiableCollection(newGroups));
    }

    @Override
    public void removeEntryPoints(Collection<ISymbolGroup> eps) {
        Collection<ISymbolGroup> groups = project.getHint(KEY_SYMBOL_GROUPS);
        Collection<ISymbolGroup> newGroups = new HashSet<ISymbolGroup>(groups);
        newGroups.removeAll(eps);
        project.setHint(KEY_SYMBOL_GROUPS, Collections.unmodifiableCollection(newGroups));
    }

}

public class SymbolManagerFeature extends AbstractProjectFeature {

    SymbolManager sm;

    @Override
    public void configure() throws ProjectException {
        // IMPORTANT: This is vital for making sure that certain basic
        // resource classes are loaded into the database session as its
        // services (see org.simantics.db.ServiceLocator).
        try {
            BasicResources.getInstance(getSession());
        } catch (DatabaseException e) {
            throw new ProjectException(e);
        }

        sm = new SymbolManager(getProject());
        getProjectElement().setHint(ISymbolManager.KEY_SYMBOL_MANAGER, sm);
    }

    @Override
    public void deconfigure() throws ProjectException {
        if (getProjectElement().getHint(ISymbolManager.KEY_SYMBOL_MANAGER) == sm)
            getProjectElement().removeHint(ISymbolManager.KEY_SYMBOL_MANAGER);
        sm.dispose();
        sm = null;
    }

}