/*******************************************************************************
 * Copyright (c) 2007, 2024 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *     Semantum Oy - GitLab #1070
 *******************************************************************************/
package org.simantics.diagram.symbolcontribution;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Objects;

import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.common.request.BinaryRead;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.adapter.Instances;
import org.simantics.db.service.CollectionSupport;
import org.simantics.diagram.stubs.DiagramResource;
import org.simantics.diagram.symbollibrary.ISymbolGroup;
import org.simantics.layer0.Layer0;

/**
 * @author Antti Villberg
 * @author Tuukka Lehtonen
 */
public class IndexRootSymbolProviderFactory implements SymbolProviderFactory {

    private final Resource indexRoot;
    private final Resource diagram;
    private byte[] contributionHash;
    private transient int hash;

    public IndexRootSymbolProviderFactory(ReadGraph graph, Resource indexRoot, Resource diagram) throws DatabaseException {
        this.indexRoot = indexRoot;
        this.diagram = diagram;
        // This is used for hashCode/equals, not diagram to:
        // 1. avoid separate symbol library page for every diagram editor
        // 2. still take into account symbol contribution filters that perform checks on diagram instance
        this.contributionHash = graph.syncRequest(new LoadHashRequest(indexRoot, diagram));
    }

    private int hash() {
        final int prime = 31;
        int result = 1;
        result = prime * result + ((indexRoot == null) ? 0 : indexRoot.hashCode());
        //result = prime * result + ((diagram == null) ? 0 : diagram.hashCode());
        result = prime * result + Arrays.hashCode(contributionHash);
        return result;
    }

    @Override
    public int hashCode() {
        if (hash == 0) {
            hash = hash();
        }
        return hash;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj)
            return true;
        if (obj == null)
            return false;
        if (getClass() != obj.getClass())
            return false;
        IndexRootSymbolProviderFactory other = (IndexRootSymbolProviderFactory) obj;
        return 
                Objects.equals(indexRoot, other.indexRoot) &&
                //Objects.equals(diagram, other.diagram) &&
                Arrays.equals(contributionHash, other.contributionHash)
                ;
    }

    @Override
    public ISymbolProvider create(ReadGraph g) throws DatabaseException {
        return g.syncRequest(new LoadRequest(indexRoot, diagram));
    }

    static class LoadHashRequest extends BinaryRead<Resource, Resource, byte[]> {
        public LoadHashRequest(Resource indexRoot, Resource diagram) {
            super(indexRoot, diagram);
        }
        @Override
        public byte[] perform(ReadGraph graph) throws DatabaseException {
            Layer0 L0 = Layer0.getInstance(graph);
            DiagramResource DIA = DiagramResource.getInstance(graph);

            CollectionSupport cs = graph.getService(CollectionSupport.class);
            Hasher h = new Hasher();

            if (parameter != null) {
                h.update(parameter);
                Instances query = graph.adapt(DIA.SymbolReferenceLibrary, Instances.class);
                for (Resource library : cs.asSortedList(query.find(graph, parameter))) {
                    BasicSymbolProviderFactory.hashGroup(graph, h, library, L0.IsRelatedTo, parameter2);
                }
            }

            return h.digest();
        }
    }

    static class LoadRequest extends BinaryRead<Resource, Resource, ISymbolProvider> {
        public LoadRequest(Resource indexRoot, Resource diagram) {
            super(indexRoot, diagram);
        }
        @Override
        public ISymbolProvider perform(ReadGraph graph) throws DatabaseException {
            Layer0 L0 = graph.l0();
            DiagramResource DIA = DiagramResource.getInstance(graph);

            var groups = new ArrayList<ISymbolGroup>();
            CollectionSupport cs = graph.getService(CollectionSupport.class);

            if (parameter != null) {
                Instances query = graph.adapt(DIA.SymbolReferenceLibrary, Instances.class);
                for (Resource library : cs.asSortedList(query.find(graph, parameter))) {
                    if (SymbolProviderFactories.accept(graph, DIA, library, parameter2)) {
                        groups.add(BasicSymbolProviderFactory.createGroup(graph, library, L0.IsRelatedTo, parameter2));
                    }
                }
            }

            return new SymbolProvider(groups);
        }
    }

}
