/*******************************************************************************
 * Copyright (c) 2007, 2024 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *     Semantum Oy - GitLab #1070
 *******************************************************************************/
package org.simantics.diagram.symbolcontribution;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Objects;

import org.simantics.databoard.Bindings;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.common.request.BinaryRead;
import org.simantics.db.common.request.ObjectsWithSupertype;
import org.simantics.db.common.utils.NameUtils;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.adapter.Instances;
import org.simantics.db.service.CollectionSupport;
import org.simantics.diagram.stubs.DiagramResource;
import org.simantics.diagram.symbollibrary.ISymbolGroup;
import org.simantics.diagram.symbollibrary.ISymbolItem;
import org.simantics.layer0.Layer0;
import org.simantics.utils.strings.AlphanumComparator;

/**
 * A basic SymbolProviderFactory implementation for the graph database. It is
 * capable of loading an ISymbolProvider from BasicSymbolContribution instances
 * like the following:
 * 
 * <pre>
 * _ : DIA.BasicSymbolContribution
 *     DIA.BasicSymbolContributionHasSymbolLibrary
 *         MySymbolLibrary1
 *         MySymbolLibrary2
 * </pre>
 * 
 * @author Tuukka Lehtonen
 */
public class BasicSymbolProviderFactory implements SymbolProviderFactory {

    private final Resource contribution;
    private final Resource diagram;
    private byte[] contributionHash;
    private transient int hash;

    public BasicSymbolProviderFactory(ReadGraph graph, Resource contribution, Resource diagram) throws DatabaseException {
        this.contribution = contribution;
        this.diagram = diagram;
        // This is used for hashCode/equals, not diagram to:
        // 1. avoid separate symbol library page for every diagram editor
        // 2. still take into account symbol contribution filters that perform checks on diagram instance
        this.contributionHash = graph.syncRequest(new LoadHashRequest(contribution, diagram));
    }

    private int hash() {
        final int prime = 31;
        int result = 1;
        result = prime * result + Objects.hashCode(contribution);
        //result = prime * result + Objects.hashCode(diagram);
        result = prime * result + Arrays.hashCode(contributionHash);
        return result;
    }

    @Override
    public int hashCode() {
        if (hash == 0) {
            hash = hash();
        }
        return hash;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj)
            return true;
        if (obj == null)
            return false;
        if (getClass() != obj.getClass())
            return false;
        BasicSymbolProviderFactory other = (BasicSymbolProviderFactory) obj;
        return 
                Objects.equals(contribution, other.contribution) &&
                //Objects.equals(diagram, other.diagram) &&
                Arrays.equals(contributionHash, other.contributionHash)
                ;
    }

    @Override
    public ISymbolProvider create(ReadGraph g) throws DatabaseException {
        return g.syncRequest(new LoadRequest(contribution, diagram));
    }

    static class LoadHashRequest extends BinaryRead<Resource, Resource, byte[]> {
        public LoadHashRequest(Resource contribution, Resource diagram) {
            super(contribution, diagram);
        }
        @Override
        public byte[] perform(ReadGraph graph) throws DatabaseException {
            Layer0 L0 = graph.l0();
            DiagramResource DIA = DiagramResource.getInstance(graph);

            CollectionSupport cs = graph.getService(CollectionSupport.class);
            Hasher h = new Hasher();

            if (SymbolProviderFactories.accept(graph, DIA, parameter, parameter2)) {
                h.update(parameter);
                Instances query = graph.adapt(DIA.SymbolReferenceLibrary, Instances.class);
                for (Resource library : cs.asSortedList(query.find(graph, parameter))) {
                    hashGroup(graph, h, library, L0.DependsOn, parameter2);
                }
            }

            return h.digest();
        }
    }

    /*
     * Note: this cannot be ResourceRead since it must never be
     * classified as immutable because of possible dynamic filters
     */
    static class LoadRequest extends BinaryRead<Resource, Resource, ISymbolProvider> {
        public LoadRequest(Resource contribution, Resource diagram) {
            super(contribution, diagram);
        }
        @Override
        public ISymbolProvider perform(ReadGraph graph) throws DatabaseException {
            Layer0 L0 = Layer0.getInstance(graph);
            DiagramResource DIA = DiagramResource.getInstance(graph);

            if (!SymbolProviderFactories.accept(graph, DIA, parameter, parameter2))
                return new SymbolProvider(Collections.<ISymbolGroup>emptyList());

            Collection<Resource> libraries = graph.getObjects(parameter, DIA.BasicSymbolContributionHasSymbolLibrary);
            var groups = new ArrayList<ISymbolGroup>(libraries.size());

            for (Resource library : libraries) {
                if (SymbolProviderFactories.accept(graph, DIA, library, parameter2)) {
                    groups.add(createGroup(graph, library, L0.DependsOn, parameter2));
                }
            }

            return new SymbolProvider(groups);
        }
    }

    static Hasher hashGroup(ReadGraph graph, Hasher hasher, Resource library, Resource relation, Resource diagram) throws DatabaseException {
        DiagramResource DIA = DiagramResource.getInstance(graph);
        if (!SymbolProviderFactories.accept(graph, DIA, library, diagram))
            return hasher;

        hasher.update(library);

        CollectionSupport cs = graph.getService(CollectionSupport.class);
        for (Resource item : cs.asSortedList(graph.syncRequest(new ObjectsWithSupertype(library, relation, DIA.Element)))) {
            if (SymbolProviderFactories.accept(graph, DIA, item, diagram)) {
                hasher.update(item);
            }
        }

        return hasher;
    }

    static SymbolGroup createGroup(ReadGraph graph, Resource library, Resource relation, Resource diagram) throws DatabaseException {
        Layer0 L0 = graph.l0();
        DiagramResource DIA = DiagramResource.getInstance(graph);

        String name = safeName(graph, library);
        String description = graph.getPossibleRelatedValue2(library, L0.HasDescription, Bindings.STRING);
        if (description == null)
            description = name;
        ModifiableSymbolGroup group = new ModifiableSymbolGroup(library, name, description);

        var items = new ArrayList<ISymbolItem>();
        for (Resource item : graph.syncRequest(new ObjectsWithSupertype(library, relation, DIA.Element))) {
            if (!SymbolProviderFactories.accept(graph, DIA, item, diagram)) {
                continue;
            }

            String itemName = safeName(graph, item);
            String itemDescription = graph.getPossibleRelatedValue2(item, L0.HasDescription, Bindings.STRING);
            if (itemDescription == null || itemDescription.isEmpty())
                itemDescription = itemName;
            items.add( new ElementSymbolItem(item, itemName, itemDescription, group) );
        }

        Collections.sort(items, (ISymbolItem o1, ISymbolItem o2) -> {
            return AlphanumComparator.CASE_INSENSITIVE_COMPARATOR.compare(o1.getName(), o2.getName());
        });

        group.setItems(items.toArray(new ISymbolItem[items.size()]));

        return group;
    }

    private static String safeName(ReadGraph graph, Resource r) throws DatabaseException {
        Layer0 L0 = graph.l0();
        String name = graph.getPossibleRelatedValue2(r, L0.HasLabel, Bindings.STRING);
        if (name == null || name.isEmpty())
            name = graph.getPossibleRelatedValue(r, L0.HasName, Bindings.STRING);
        if (name == null)
            name = NameUtils.getSafeName(graph, r);
        return name;
    }

}
