/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.diagram.symbolcontribution;

import org.eclipse.core.runtime.IAdaptable;

/**
 * A basic pair structure that implements IAdaptable. It tries to adapt the
 * paired objects to the requested class in order (first, second).
 * 
 * @author Tuukka Lehtonen
 * 
 * @param <T1> type of first
 * @param <T2> type of second
 */
final class AdaptablePair<T1, T2> implements IAdaptable {
    public final T1 first;
    public final T2 second;
    private final int hash;

    public static <T1, T2> AdaptablePair<T1, T2> make(T1 t1, T2 t2) {
        return new AdaptablePair<T1, T2>(t1, t2);
    }

    public AdaptablePair(T1 first, T2 second) {
        this.first = first;
        this.second = second;
        this.hash = makeHash();
    }

    @Override
    public boolean equals(Object obj) {
        if (obj == null)
            return false;
        if (!(obj.getClass().equals(this.getClass())))
            return false;
        AdaptablePair<?, ?> other = (AdaptablePair<?, ?>) obj;
        if (other.first != first && (other.first == null || !other.first.equals(first)))
            return false;
        if (other.second != second && (other.second == null || !other.second.equals(second)))
            return false;
        return true;
    }

    @Override
    public int hashCode() {
        return hash;
    }

    @Override
    public String toString() {
        return "<"+first+", "+second+">";
    }

    private int makeHash() {
        return (first == null ? 0 : first.hashCode()) + (second == null ? 0 : second.hashCode())*31;
    }

    private Object adapt(Object o, Class<?> clazz) {
        if (clazz.isInstance(o))
            return o;
        if (o instanceof IAdaptable)
            return ((IAdaptable) o).getAdapter(clazz);
        return null;
    }

    @SuppressWarnings("rawtypes")
    @Override
    public Object getAdapter(Class adapter) {
        Object o = adapt(first, adapter);
        return (o == null) ? adapt(second, adapter) : o;
    }

}
