/*******************************************************************************
 * Copyright (c) 2010, 2013 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *     Semantum Oy - issue #4384
 *******************************************************************************/
package org.simantics.diagram.runtime;

import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.common.request.TernaryRead;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.request.PossibleActiveRun;
import org.simantics.db.layer0.variable.RVI;
import org.simantics.db.layer0.variable.Variable;
import org.simantics.db.layer0.variable.Variables;
import org.simantics.diagram.stubs.DiagramResource;
import org.simantics.layer0.Layer0;
import org.simantics.modeling.ModelingResources;

/**
 * Extract a triple <Model URI, Variable URI, RVI> from the input arguments if
 * possible.
 * 
 * Returns <code>null</code> if it cannot find:
 * <ol>
 * <li>model resource by its URI</li>
 * <li>active realization of the model or base realization as a fallback</li>
 * </ol>
 * 
 * @author Tuukka Lehtonen
 */
class RuntimeVariable extends TernaryRead<Resource, RVI, Resource, RuntimeDiagramDesc> {

    public RuntimeVariable(Resource model, RVI rvi, Resource diagramResource) {
        super(model, rvi, diagramResource);
    }

    @Override
    public RuntimeDiagramDesc perform(ReadGraph graph) throws DatabaseException {

        Resource model = parameter;
        RVI rvi = parameter2;
        Resource diagram = parameter3;

        Variable variable = null;
        String runtimeProfileURI = null;

        if (model != null && rvi != null) {
        	Layer0 L0 = Layer0.getInstance(graph);
        	if (graph.isInstanceOf(model, L0.IndexRoot)) {
        		Variable activeRun = graph.syncRequest(new PossibleActiveRun(model));
        		Variable context = activeRun != null ? activeRun : Variables.getPossibleConfigurationContext(graph, model); 
        		if (context != null) variable = rvi.resolvePossible(graph, context);
        	}

        } else {

        	Resource composite = graph.getPossibleObject(diagram, ModelingResources.getInstance(graph).DiagramToComposite);
        	if (composite != null) {
        		variable = Variables.getPossibleVariable(graph, composite);
        	} else {
        		variable = Variables.getPossibleVariable(graph, diagram);
        	}

        }

        if (model != null) {
            Resource runtimeProfile = getActiveProfile(graph, model, diagram);
            if (runtimeProfile != null) {
                runtimeProfileURI = graph.getPossibleURI(runtimeProfile);
            }
        }

        return new RuntimeDiagramDesc(makeModelURI(graph, variable, model), makeVariableURI(graph, variable), makeRVIString(graph, variable, rvi), runtimeProfileURI);

    }

    String makeModelURI(ReadGraph graph, Variable variable, Resource model) throws DatabaseException {
        if (variable != null) {
            Resource m = Variables.getPossibleIndexRoot(graph, variable);
            return m != null ? graph.getPossibleURI(m) : null;
        }
        return model != null ? graph.getPossibleURI(model) : null;
    }

    String makeVariableURI(ReadGraph graph, Variable variable) throws DatabaseException {
        return variable != null ? variable.getURI(graph) : null;
    }

    String makeRVIString(ReadGraph graph, Variable variable, RVI rvi) throws DatabaseException {
        if(variable != null) {
        	RVI r = variable.getPossibleRVI(graph); 
        	if(r != null) return r.toString();
        }
        return rvi != null ? rvi.toString() : null;
    }

    private static Resource getActiveProfile(ReadGraph graph, Resource model, Resource diagram) throws DatabaseException {
        DiagramResource DIA = DiagramResource.getInstance(graph);
        // This has been disabled to make diagram profile selection model-specific instead of diagram-specific.
//        Resource activeProfile = graph.getPossibleObject(diagram, DIA.HasActiveProfile);
//        if(activeProfile != null) return activeProfile;
        return graph.getPossibleObject(model, DIA.HasActiveProfile);
    }

}