/*******************************************************************************
 * Copyright (c) 2007, 2013 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.diagram.participant;

import java.awt.event.FocusEvent;
import java.awt.event.FocusListener;

import org.simantics.g2d.canvas.ICanvasContext;
import org.simantics.g2d.canvas.impl.AbstractCanvasParticipant;
import org.simantics.g2d.canvas.impl.DependencyReflection.Reference;
import org.simantics.g2d.chassis.AWTChassis;
import org.simantics.g2d.chassis.SWTChassis;

/**
 * The purpose of this participant is to properly handle contexts in Eclipse/SWT
 * based deployments G2D canvases where it is possible that Eclipse eats input
 * events based on UI contexts, key bindings, etc.
 * 
 * This participant depends on the {@link ContextUtil} canvas participant.
 * 
 * When some other scene graph component or UI Part, besides the root pane, gets focus,
 * this participant will deactivate its {@link #contextId}. When the root pane gets the 
 * focus back, it will restore {@link #contextId} to allow Eclipse key bindings
 * for that {@link #contextId} to work properly. 
 * 
 * @see ContextUtil
 * 
 * @author J-P Laine
 * @author Tuukka Lehtonen
 * @author Teemu Lempinen
 */
public class SGFocusParticipant extends AbstractCanvasParticipant implements FocusListener {

    @Reference
    ContextUtil                 contextUtil;
    
    @Reference
    org.simantics.diagram.participant.e4.ContextUtil e4ContextUtil;

    final SWTChassis            chassis;
    String                      contextId;
    ContextState                originalContextState;

    public SGFocusParticipant(SWTChassis chassis, String contextId) {
        super();
        this.chassis = chassis;
        this.contextId = contextId;
    }

    @Override
    public void addedToContext(ICanvasContext ctx) {
        super.addedToContext(ctx);
        chassis.getAWTComponent().addFocusListener(this);
    }

    @Override
    public void removedFromContext(ICanvasContext ctx) {
        if (!chassis.isDisposed()) {
            AWTChassis component = chassis.getAWTComponent();
            if (component != null)
                component.removeFocusListener(this);
        }
        super.removedFromContext(ctx);
    }

    @Override
    public void focusGained(FocusEvent e) {
        ContextUtil util = contextUtil;
        if (util != null) {
            DiagramCommandBindings.activateBindings(util, contextId);
        } else {
            org.simantics.diagram.participant.e4.ContextUtil e4Util = e4ContextUtil;
            if (e4Util != null) {
                DiagramCommandBindings.activateBindings(e4Util, contextId);
            }
        }
    }

    @Override
    public void focusLost(FocusEvent e) {
        ContextUtil util = contextUtil;
        if (util != null) {
            DiagramCommandBindings.deactivateBindings(util, contextId);
        } else {
            org.simantics.diagram.participant.e4.ContextUtil e4Util = e4ContextUtil;
            if (e4Util != null) {
                DiagramCommandBindings.deactivateBindings(e4Util, contextId);
            }
        }
    }

}