/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.diagram.handler;

import java.awt.geom.Point2D;
import java.util.Collections;
import java.util.Map;

import org.simantics.db.Resource;
import org.simantics.g2d.canvas.ICanvasContext;
import org.simantics.g2d.diagram.IDiagram;
import org.simantics.scenegraph.g2d.events.command.Command;

/**
 * 
 * @author Tuukka Lehtonen
 */
public final class PasteOperation {

    /**
     * Optional specifications for the paste operation that can be given through.
     */
    public static interface PasteOption {}

    /**
     * For forcing pasting of diagram reference elements even when the parent
     * element of the reference element is not included in the copied material.
     */
    public static enum ForceCopyReferences implements PasteOption { INSTANCE }

    // Input
    public final Command                 command;
    public final ICanvasContext          ctx;
    public final Resource                sourceDiagram;
    public final Resource                targetDiagram;
    public final IDiagram                target;
    public final ElementObjectAssortment ea;
    public final boolean                 cut;
    public final Point2D                 offset;

    public Map<Resource, Resource>       initialNodeMap;

    public PasteOption[]                 options;

    // Output
    public final Map<Object, Object>     copyMap;

    public PasteOperation(
            Command command,
            ICanvasContext ctx,
            Resource sourceDiagram,
            Resource targetDiagram,
            IDiagram target,
            ElementObjectAssortment ea,
            boolean cut,
            Point2D offset,
            Map<Resource, Resource> initialNodeMap,
            Map<Object, Object> copyMap
    ) {
        if (sourceDiagram == null)
            throw new IllegalArgumentException("null source diagram");
        if (targetDiagram == null)
            throw new IllegalArgumentException("null target diagram");

        this.command = command;
        this.ctx = ctx;
        this.sourceDiagram = sourceDiagram;
        this.targetDiagram = targetDiagram;
        this.target = target;
        this.ea = ea;
        this.cut = cut;
        this.offset = offset;
        this.copyMap = copyMap;
        this.initialNodeMap = initialNodeMap;
    }

    public PasteOperation(
            Command command,
            ICanvasContext ctx,
            Resource sourceDiagram,
            Resource targetDiagram,
            IDiagram target,
            ElementObjectAssortment ea,
            boolean cut,
            Point2D offset,
            Map<Resource, Resource> initialNodeMap
    ) {
        this(command, ctx, sourceDiagram, targetDiagram, target, ea, cut, offset, initialNodeMap, null);
    }

    public PasteOperation(
            Command command,
            ICanvasContext ctx,
            Resource sourceDiagram,
            Resource targetDiagram,
            IDiagram target,
            ElementObjectAssortment ea,
            boolean cut,
            Point2D offset
    ) {
        this(command, ctx, sourceDiagram, targetDiagram, target, ea, cut, offset, Collections.<Resource, Resource>emptyMap());
    }

    public PasteOperation options(PasteOption... options) {
        this.options = options;
        return this;
    }

    public boolean sameDiagram() {
        return sourceDiagram.equals(targetDiagram);
    }

    @SuppressWarnings("unchecked")
    public <T extends PasteOption> T getOption(Class<T> clazz) {
        if (options == null)
            return null;
        for (PasteOption option : options) {
            if (clazz.isInstance(option))
                return (T) option;
        }
        return null;
    }

    public <T extends PasteOption> boolean hasOption(Class<T> clazz) {
        return getOption(clazz) != null;
    }

}
