/*******************************************************************************
 * Copyright (c) 2024 Association for Decentralized Information Management in
 * Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Semantum Oy - initial API and implementation
 *******************************************************************************/

package org.simantics.diagram.handler;

import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Deque;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import org.eclipse.jface.action.Action;
import org.eclipse.jface.action.ActionContributionItem;
import org.eclipse.jface.action.ContributionItem;
import org.eclipse.jface.action.IAction;
import org.eclipse.jface.action.IContributionItem;
import org.eclipse.jface.dialogs.IInputValidator;
import org.eclipse.jface.dialogs.InputDialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.swt.widgets.MenuItem;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.internal.handlers.WizardHandler.New;
import org.simantics.Simantics;
import org.simantics.databoard.Bindings;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.WriteGraph;
import org.simantics.db.common.request.WriteRequest;
import org.simantics.db.common.utils.NameUtils;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.util.Layer0Utils;
import org.simantics.diagram.stubs.DiagramResource;
import org.simantics.layer0.Layer0;
import org.simantics.modeling.ModelingResources;
import org.simantics.structural.stubs.StructuralResource2;
import org.simantics.ui.contribution.DynamicMenuContribution;
import org.simantics.ui.workbench.IResourceEditorInput;
import org.simantics.utils.ui.AdaptionUtils;

public class ConnectionCrossingMenuContribution extends DynamicMenuContribution{
    
    private static final IContributionItem EMPTY_ITEM = new ContributionItem() {
       @Override
        public void fill(final Menu menu, int index) {
            MenuItem crossings = new MenuItem(menu, SWT.CASCADE, index);
            crossings.setText("Connection Crossings");
            Menu crossingsMenu = new Menu(menu);
            crossings.setMenu(crossingsMenu);
            crossingsMenu.setEnabled(false);
            crossings.setEnabled(false);
        }
        
    };
    
    private static final IContributionItem[] NO_ITEMS = {EMPTY_ITEM};
    
    
    private static List<Resource> getDiagramForSelection(ReadGraph graph, Resource r) throws DatabaseException{
        Layer0 L0 = Layer0.getInstance(graph);
        DiagramResource DIA = DiagramResource.getInstance(graph);
        StructuralResource2 STR = StructuralResource2.getInstance(graph);
        ModelingResources MOD = ModelingResources.getInstance(graph);
        
        List<Resource> result = new ArrayList<Resource>();
        Deque<Resource> stack = new ArrayDeque<Resource>();
        if (r != null)
            stack.push(r);
        while (!stack.isEmpty()) {
            r = stack.pop();
            if (graph.isInstanceOf(r, DIA.Diagram))
                result.add(r);
            else if (graph.isInstanceOf(r, STR.Composite)) {
                Resource diagram = graph.getPossibleObject(r, MOD.CompositeToDiagram);
                if (diagram != null) {
                    // Composite is paired to a diagram.
                    result.add(diagram);
                } else {
                    // Composite is a "folder".
                    stack.addAll(graph.getObjects(r, L0.ConsistsOf));
                }
            } else if (graph.isInstanceOf(r, L0.Library)) {
                // Generic fallback. This is not required at least by A6.
                stack.addAll(graph.getObjects(r, L0.ConsistsOf));
            }
        }
        return result;
    }
    
    @Override
    protected IContributionItem[] getContributionItems(ReadGraph graph, Object[] selection) throws DatabaseException {
        
        DiagramResource DIA = DiagramResource.getInstance(graph);
        
        List<Resource> diagrams = new ArrayList<Resource>();
        if (selection.length >= 1) {
            for (Object sel : selection) {
                Resource r = AdaptionUtils.adaptToSingle(sel, Resource.class);
                if (r == null)
                    return NO_ITEMS;
                List<Resource> diags = getDiagramForSelection(graph, r);
                if (diags.size() == 0) {
                    return NO_ITEMS;
                }
                diagrams.addAll(diags);
            }
        }
        if (diagrams.size() == 0) {
            // Attempt to get diagram Resource from the active editor.
            IWorkbenchWindow window = PlatformUI.getWorkbench().getActiveWorkbenchWindow();
            IEditorPart editorPart = null;
            if (window != null) {
                editorPart =  window.getActivePage().getActiveEditor();
            } else {
                for (IWorkbenchWindow w : PlatformUI.getWorkbench().getWorkbenchWindows()) {
                    if (w.getActivePage() != null) {
                        editorPart =  w.getActivePage().getActiveEditor();
                        if (editorPart != null)
                            break;
                    }
                }
            }
            if (editorPart == null)
                return NO_ITEMS;
            IEditorInput input = editorPart.getEditorInput();
            if (!(input instanceof IResourceEditorInput))
                return NO_ITEMS;
            Resource diagram = ((IResourceEditorInput)input).getResource();
            if (diagram != null)
                diagrams.add(diagram);
        }
        if (diagrams.size() == 0)
            return NO_ITEMS;
        Map<Resource, Integer> currentTypes = new HashMap<Resource, Integer>();
        List<Double> currentWidths = new ArrayList<Double>();
        for (Resource diagram : diagrams) {
            Resource currentType = graph.getPossibleObject(diagram, DIA.ConnectionCrossingStyle_HasType);
            if (currentType == null) {
                continue;
            }
            Integer c = currentTypes.get(currentType);
            if (c == null) {
                c = 1;
            } else {
                c++;
            }
            currentTypes.put(currentType, c);
            
            Double width = graph.getPossibleRelatedValue(diagram, DIA.ConnectionCrossingStyle_Width, Bindings.DOUBLE);
            if (width != null)
                currentWidths.add(width);
        }
        if (currentTypes.size() == 0)
            // We assume that a diagram must have at least asserted crossing type.
            return NO_ITEMS;
        Resource currentType = null;
        Double currentWidth = null;
        {
            int wc = 0;
            for (Entry<Resource, Integer> e : currentTypes.entrySet()) {
                if (wc < e.getValue()) {
                    wc = e.getValue();
                    currentType = e.getKey();
                }
            }
            if (currentWidths.size() == 0) {
                
            } else if (currentWidths.size() == 1) {
                currentWidth = currentWidths.get(0);
            } else {
                Collections.sort(currentWidths);
                int mid = currentWidths.size() /2;
                currentWidth = currentWidths.get(mid);
            }
        }
        
        
        List<Resource> crossingTypes = new ArrayList<Resource>(4);
        crossingTypes.add(DIA.ConnectionCrossingStyle_Type_None);
        crossingTypes.add(DIA.ConnectionCrossingStyle_Type_Arc);
        crossingTypes.add(DIA.ConnectionCrossingStyle_Type_Gap);
        crossingTypes.add(DIA.ConnectionCrossingStyle_Type_Square);
        
        List<IAction> actions = new ArrayList<IAction>(crossingTypes.size()+2);
        for (Resource ct : crossingTypes) {
            String name = NameUtils.getSafeName(graph, ct);
            SetCrossingTypeAction act = new SetCrossingTypeAction(name, diagrams, ct);
            actions.add(act);
            if (ct.equals(currentType)) {
                act.setChecked(true);
            }
        }
        if (currentWidth != null) {
            if (currentWidth > 0.0) {
                double nw = currentWidth - 0.5;
                if (nw < 0.0)
                    nw = 0.0;
                actions.add(new SetCrossingWidthAction(diagrams, nw, false));
            }
            {
                double nw = currentWidth + 0.5;
                if (nw > 0.0) {
                    actions.add(new SetCrossingWidthAction(diagrams, nw, false));
                }
            }
            actions.add(new SetCrossingWidthAction(diagrams, currentWidth, true));
        } else {
            actions.add(new SetCrossingWidthAction(diagrams, 1.0, true));
        }
        return new IContributionItem[] { new ContributionItem() {
            @Override
            public void fill(final Menu menu, int index) {
                MenuItem crossings = new MenuItem(menu, SWT.CASCADE, index);
                crossings.setText("Connection Crossings");
                Menu crossingsMenu = new Menu(menu);
                crossings.setMenu(crossingsMenu);
                int i = 0;
                for (IAction act : actions) {
                    ActionContributionItem item = new ActionContributionItem(act);
                    item.fill(crossingsMenu, i);
                    i++;
                }
            }
        }};
    }

    
    private static class SetCrossingTypeAction extends Action {
        private List<Resource> diagrams;
        private Resource crossingType;
        
        public SetCrossingTypeAction(String text, List<Resource> diagrams, Resource crossingType) {
            super(text, AS_RADIO_BUTTON);
            this.diagrams = diagrams;
            this.crossingType = crossingType;
        }

        @Override
        public void run() {
            Simantics.getSession().asyncRequest(new WriteRequest() {
                
                @Override
                public void perform(WriteGraph graph) throws DatabaseException {
                    DiagramResource DIA = DiagramResource.getInstance(graph);
                    for (Resource diagram : diagrams) {
                        graph.deny(diagram,DIA.ConnectionCrossingStyle_HasType);
                        graph.claim(diagram, DIA.ConnectionCrossingStyle_HasType, crossingType);
                    }
                    Layer0Utils.addCommentMetadata(graph, "Set crossing type to " + getText());
                    graph.markUndoPoint();
                }
            });
        }
        
    }
    
    private static class SetCrossingWidthAction extends Action {
        private List<Resource> diagrams;
        private Double width;
        private Double currentWidth;
        
        public SetCrossingWidthAction(List<Resource> diagrams, double width, boolean current) {
            super(current ? "Set width... " : "Set width " + width, AS_PUSH_BUTTON);
            this.diagrams = diagrams;
            if (current)
                this.currentWidth = width;
            else
                this.width = width;
        }
        
        @Override
        public void run() {
            if (width == null) {
                IInputValidator validator = new IInputValidator() {
                    
                    @Override
                    public String isValid(String newText) {
                        try {
                            double d = Double.parseDouble(newText);
                            if (d < 0.0)
                                return "Value must be positive.";
                        } catch (NumberFormatException e) {
                            return "Input is not a number.";
                        }
                        return null;
                    }
                };
                InputDialog dialog = new InputDialog(Display.getCurrent().getActiveShell() , "Diagram", "Set crossing width", currentWidth.toString(), validator);
                if (dialog.open() != InputDialog.OK)
                    return;
                width = Double.parseDouble(dialog.getValue());
                
            }
            if (width == null)
                return;
            Simantics.getSession().asyncRequest(new WriteRequest() {
                
                @Override
                public void perform(WriteGraph graph) throws DatabaseException {
                    DiagramResource DIA = DiagramResource.getInstance(graph);
                    for (Resource diagram : diagrams) {
                        graph.deny(diagram,DIA.ConnectionCrossingStyle_Width);
                        graph.claimLiteral(diagram, DIA.ConnectionCrossingStyle_Width, width, Bindings.DOUBLE);
                    }
                    Layer0Utils.addCommentMetadata(graph, "Set crossing width");
                    graph.markUndoPoint();
                }
            });
        }
    }
}
