/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.diagram.content;

import java.util.HashSet;
import java.util.Set;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.atomic.AtomicInteger;

import org.simantics.db.AsyncReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.common.request.ResourceAsyncRead;
import org.simantics.db.procedure.AsyncMultiProcedure;
import org.simantics.db.procedure.AsyncProcedure;
import org.simantics.diagram.stubs.DiagramResource;
import org.simantics.diagram.synchronization.ErrorHandler;

/**
 * @author Antti Villberg
 * @author Tuukka Lehtonen
 */
public class RouteGraphConnectionPartRequest extends ResourceAsyncRead<RouteGraphConnectionPartData> {

    final ErrorHandler    errorHandler;
    final DiagramResource DIA;

    public RouteGraphConnectionPartRequest(ErrorHandler errorHandler, DiagramResource dia, Resource resource) {
        super(resource);
        this.errorHandler = errorHandler;
        this.DIA = dia;
    }

    @Override
    public void perform(AsyncReadGraph graph, final AsyncProcedure<RouteGraphConnectionPartData> procedure) {

        final AtomicInteger ready = new AtomicInteger(3);
        final RouteGraphConnectionPartData result = new RouteGraphConnectionPartData();
        final HashSet<EdgeResource> connectorLinks = new HashSet<EdgeResource>();
        final HashSet<EdgeResource> routeLinks = new HashSet<EdgeResource>();

        result.links = connectorLinks;
        result.routeLines = new ConcurrentLinkedQueue<Resource>();
        result.routePoints = new ConcurrentLinkedQueue<Resource>();

        AsyncMultiProcedure<Resource> locationProcedure = new AsyncMultiProcedure<Resource>() {

            @Override
            public void exception(AsyncReadGraph graph, Throwable throwable) {
                errorHandler.error("forEachObject(connection, Has I/O) failure", throwable);
            }

            @Override
            public void execute(AsyncReadGraph graph, final Resource location) {

                ready.incrementAndGet();

                graph.forEachObject(location, DIA.AreConnected, new AsyncMultiProcedure<Resource>() {

                    @Override
                    public void exception(AsyncReadGraph graph, Throwable throwable) {
                        errorHandler.error("forEachObject(:I/O Connector, Are Connected) failure", throwable);
                    }

                    @Override
                    public void execute(AsyncReadGraph graph, Resource connectedTo) {
                        synchronized (connectorLinks) {
                            connectorLinks.add(new EdgeResource(location, connectedTo));
                        }
                    }

                    @Override
                    public void finished(AsyncReadGraph graph) {
                        if (ready.decrementAndGet() == 0) {
                            connectorLinks.addAll(routeLinks);
                            procedure.execute(graph, result);
                        }
                    }

                });

            }

            @Override
            public void finished(AsyncReadGraph graph) {
                if (ready.decrementAndGet() == 0) {
                    connectorLinks.addAll(routeLinks);
                    procedure.execute(graph, result);
                }
            }

        };

        AsyncMultiProcedure<Resource> locationProcedure2 = new AsyncMultiProcedure<Resource>() {

            @Override
            public void exception(AsyncReadGraph graph, Throwable throwable) {
                errorHandler.error("forEachObject(connection, Has Branch Point) failure", throwable);
            }

            @Override
            public void execute(AsyncReadGraph graph, final Resource location) {

                ready.incrementAndGet();

                graph.forTypes(location, new AsyncProcedure<Set<Resource>>() {
                    @Override
                    public void execute(AsyncReadGraph graph, Set<Resource> types) {
                        if (types.contains(DIA.RouteLine)) {
                            result.routeLines.add(location);
                        } else if (types.contains(DIA.RoutePoint)) {
                            result.routePoints.add(location);
                        } else {
                            errorHandler.error("unrecognized interior route node: " + location, new Exception("trace"));
                        }

                        graph.forEachObject(location, DIA.AreConnected, new AsyncMultiProcedure<Resource>() {

                            @Override
                            public void exception(AsyncReadGraph graph, Throwable throwable) {
                                errorHandler.error("forEachObject(:Branch Point, Are Connected) failure", throwable);
                            }

                            @Override
                            public void execute(AsyncReadGraph graph, Resource connectedTo) {
                                synchronized (routeLinks) {
                                    routeLinks.add(new EdgeResource(location, connectedTo));
                                }
                            }

                            @Override
                            public void finished(AsyncReadGraph graph) {
                               if (ready.decrementAndGet() == 0) {
                                    connectorLinks.addAll(routeLinks);
                                    procedure.execute(graph, result);
                                }
                            }

                        });
                    }
                    @Override
                    public void exception(AsyncReadGraph graph, Throwable throwable) {
                        errorHandler.error("forEachObject(:Branch Point, Are Connected) failure", throwable);
                    }
                });
            }

            @Override
            public void finished(AsyncReadGraph graph) {
                if (ready.decrementAndGet() == 0) {
                    connectorLinks.addAll(routeLinks);
                    procedure.execute(graph, result);
                }
            }

        };

        AsyncMultiProcedure<Resource> locationProcedure3 = new AsyncMultiProcedure<Resource>() {

            @Override
            public void exception(AsyncReadGraph graph, Throwable throwable) {
                errorHandler.error("forEachObject(connection, Has I/O) failure", throwable);
            }

            @Override
            public void execute(AsyncReadGraph graph, final Resource location) {

                ready.incrementAndGet();

                graph.forEachObject(location, DIA.AreConnected, new AsyncMultiProcedure<Resource>() {

                    @Override
                    public void exception(AsyncReadGraph graph, Throwable throwable) {
                        errorHandler.error("forEachObject(:I/O Connector, Are Connected) failure", throwable);
                    }

                    @Override
                    public void execute(AsyncReadGraph graph, Resource connectedTo) {
                        synchronized (connectorLinks) {
                            connectorLinks.add(new EdgeResource(connectedTo, location));
                        }
                    }

                    @Override
                    public void finished(AsyncReadGraph graph) {
                        if (ready.decrementAndGet() == 0) {
                            connectorLinks.addAll(routeLinks);
                            procedure.execute(graph, result);
                        }
                    }

                });

            }

            @Override
            public void finished(AsyncReadGraph graph) {
                if (ready.decrementAndGet() == 0) {
                    connectorLinks.addAll(routeLinks);
                    procedure.execute(graph, result);
                }
            }

        };

        graph.forEachObject(resource, DIA.HasPlainConnector, locationProcedure);
        graph.forEachObject(resource, DIA.HasInteriorRouteNode, locationProcedure2);
        graph.forEachObject(resource, DIA.HasArrowConnector, locationProcedure3);

    }

}
