/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.diagram.content;

import org.simantics.db.RequestProcessor;
import org.simantics.db.Resource;
import org.simantics.db.Session;
import org.simantics.db.exception.DatabaseException;
import org.simantics.diagram.adapter.DiagramContentRequest;
import org.simantics.diagram.synchronization.ErrorHandler;
import org.simantics.g2d.canvas.ICanvasContext;
import org.simantics.utils.ui.ErrorLogger;

/**
 * A utility for requesting a diagram's contents and keeping track of them and
 * obtaining differences.
 * 
 * <p>
 * This class is not thread-safe.
 * 
 * @author Tuukka Lehtonen
 */
public class DiagramContentTracker {

    private final ErrorHandler errorHandler = new ErrorHandler() {
        @Override
        public void warning(String message, Exception e) {
            ErrorLogger.defaultLogWarning(message, e);
        }
        @Override
        public void error(String message, Throwable t) {
            ErrorLogger.defaultLogError(message, t);
        }
    };

    private DiagramContents       lastContents   = new DiagramContents();
    private DiagramContentChanges lastDifference = DiagramContentChanges.EMPTY;

    private final ICanvasContext  context;
    private final Session         session;
    private final Resource        diagram;

    public static DiagramContentTracker start(ICanvasContext context, RequestProcessor processor, Resource diagram) throws DatabaseException {
        DiagramContentTracker tracker = new DiagramContentTracker(context, processor.getSession(), diagram);
        tracker.update(processor);
        return tracker;
    }

    private DiagramContentTracker(ICanvasContext context, Session session, Resource diagram) {
        if (context == null)
            throw new NullPointerException("null canvas context");
        if (session == null)
            throw new NullPointerException("null session");
        if (diagram == null)
            throw new NullPointerException("null diagram");
        this.context = context;
        this.session = session;
        this.diagram = diagram;
    }

    public DiagramContentChanges update() throws DatabaseException {
        return update(session);
    }

    public DiagramContentChanges update(RequestProcessor processor) throws DatabaseException {
        DiagramContents contents = processor.syncRequest(new DiagramContentRequest(context, diagram, errorHandler));
        DiagramContentChanges changes = contents.differenceFrom(lastContents);
        this.lastDifference = changes;
        this.lastContents = contents;
        return changes;
    }

    public DiagramContents getLastContents() {
        return lastContents;
    }

    public DiagramContentChanges getLastDifference() {
        return lastDifference;
    }

}
