/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.diagram.adapter;

import org.simantics.db.AsyncReadGraph;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.common.request.UniqueRead;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.procedure.AsyncProcedure;
import org.simantics.g2d.canvas.ICanvasContext;
import org.simantics.g2d.diagram.IDiagram;
import org.simantics.g2d.element.ElementClass;
import org.simantics.g2d.element.IElement;

/**
 * A synchronous graph request wrapper for {@link ElementFactory} and
 * {@link ElementFactoryAdapter}. Using it will provide simpler but slower
 * implementations.
 * 
 * @author Tuukka Lehtonen
 */
public class SyncElementFactory extends ElementFactoryAdapter {

    @Override
    public void create(AsyncReadGraph graph, final ICanvasContext canvas, final IDiagram diagram,
            final Resource elementType, final AsyncProcedure<ElementClass> procedure) {
        graph.asyncRequest(new UniqueRead<ElementClass>() {
            @Override
            public ElementClass perform(ReadGraph graph) throws DatabaseException {
                return create(graph, canvas, diagram, elementType);
            }
        }, procedure);
    }

    @Override
    public void load(AsyncReadGraph graph, final ICanvasContext canvas, final IDiagram diagram,
            final Resource elementResource, final IElement element, final AsyncProcedure<IElement> procedure) {
        graph.asyncRequest(new UniqueRead<IElement>() {
            @Override
            public IElement perform(ReadGraph graph) throws DatabaseException {
                if(isOk(graph, elementResource))
                    load(graph, canvas, diagram, elementResource, element);
                else
                    notOk(graph, canvas, diagram, elementResource, element);
                return element;
            }
        }, procedure);
    }

    public ElementClass create(ReadGraph graph, ICanvasContext canvas, IDiagram diagram, Resource elementType)
    throws DatabaseException {
        throw new UnsupportedOperationException("This method should be implemented by subclass " + getClass().getName());
    }
    
    public boolean isOk(ReadGraph graph, Resource elementResource) throws DatabaseException {
        return graph.hasStatement(elementResource);
    }

    public void load(ReadGraph graph, ICanvasContext canvas, IDiagram diagram, Resource elementResource, IElement element)
    throws DatabaseException {
        throw new UnsupportedOperationException("This method should be implemented by subclass " + getClass().getName());
    }

    public void notOk(ReadGraph graph, ICanvasContext canvas, IDiagram diagram, Resource elementResource, IElement element)
            throws DatabaseException {
    }

}
