/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.diagram.adapter;

import java.util.List;

import org.simantics.db.AsyncReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.common.primitiverequest.Adapter;
import org.simantics.db.common.procedure.adapter.TransientCacheAsyncListener;
import org.simantics.db.procedure.AsyncProcedure;
import org.simantics.db.procedure.Listener;
import org.simantics.diagram.synchronization.ErrorHandler;
import org.simantics.g2d.canvas.ICanvasContext;
import org.simantics.g2d.diagram.IDiagram;
import org.simantics.g2d.diagram.handler.SubstituteElementClass;
import org.simantics.g2d.element.ElementClass;
import org.simantics.g2d.element.IElement;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * @author Antti Villberg
 */
public class NodeRequest extends BaseRequest2<Resource, IElement> {

    private static final Logger LOGGER = LoggerFactory.getLogger(NodeRequest.class);

    final IDiagram diagram;
    final Listener<IElement> loadListener;

    public NodeRequest(ICanvasContext canvas, IDiagram diagram, Resource resource, Listener<IElement> loadListener) {
        super(canvas, resource);
        this.diagram = diagram;
        this.loadListener = loadListener;
    }

    @Override
    public void perform(AsyncReadGraph graph, final AsyncProcedure<IElement> procedure) {
        // Keep this code from crashing the whole DB client by unexpected
        // throwing of NPE's somewhere in the following code that leads to
        // procedure not getting called properly.
        if (diagram == null) {
            procedure.exception(graph, new NullPointerException("null diagram specified for resource " + data));
            return;
        }

//        System.out.println("NodeRequest2 " + data);
//        graph.asyncRequest(new SafeName(data), new Procedure<String>() {
//            @Override
//            public void exception(Throwable t) {
//            }
//            @Override
//            public void execute(String result) {
//                System.out.println("NodeRequest2 "  + result);
//            }
//        });

        final ErrorHandler eh = ElementFactoryUtil.getErrorHandler(diagram);

        graph.forHasStatement(data, new AsyncProcedure<Boolean>() {

            @Override
            public void exception(AsyncReadGraph graph, Throwable throwable) {
                eh.error("NodeRequest.forHasStatement failed", throwable);
                procedure.execute(graph, null);
            }

            @Override
            public void execute(AsyncReadGraph graph, Boolean result) {

                if(!result) {
                    procedure.execute(graph, null);
                    return;
                }

                graph.asyncRequest(new Adapter<ElementFactory>(data, ElementFactory.class), new TransientCacheAsyncListener<ElementFactory>() {

                    @Override
                    public void exception(AsyncReadGraph graph, Throwable throwable) {
                        eh.error("NodeRequest.asyncRequest(Adapter<ElementFactory>) failed", throwable);
                        procedure.execute(graph, null);
                    }

                    @Override
                    public void execute(AsyncReadGraph graph, final ElementFactory factory) {

//                        graph.asyncRequest(new ResourceToURI(data), new Procedure<String>() {
//                            @Override
//                            public void exception(Throwable t) {
//                            }
//                            @Override
//                            public void execute(String result) {
//                                System.out.println("NodeRequest2 factory for "  + result + " -> " + factory);
//                            }
//                        });

                        graph.asyncRequest(new GetElementClassRequest(factory, data, canvas, diagram), new TransientCacheAsyncListener<ElementClass>() {

                            @Override
                            public void exception(AsyncReadGraph graph, Throwable throwable) {
                                LOGGER.error("Unexpected error in GetElementClassRequest", throwable);
                                procedure.execute(graph, null);
                            }

                            @Override
                            public void execute(AsyncReadGraph graph, ElementClass mutableClazz) {
                                List<SubstituteElementClass> substitutes = diagram.getDiagramClass().getItemsByClass(SubstituteElementClass.class);
                                for (SubstituteElementClass subs : substitutes) {
                                    mutableClazz = subs.substitute(diagram, mutableClazz);
                                }
                                final ElementClass clazz = mutableClazz;
                                graph.asyncRequest(new SpawnRequest(canvas, clazz, data), new TransientCacheAsyncListener<IElement>() {

                                    @Override
                                    public void exception(AsyncReadGraph graph, Throwable throwable) {
                                        LOGGER.error("Unexpected error in SpawnRequest", throwable);
                                        procedure.execute(graph, null);
                                    }

                                    @Override
                                    public void execute(AsyncReadGraph graph, IElement element) {
                                        procedure.execute(graph, element);

                                        if (loadListener != null) {
                                            //System.out.println("LoadRequest[" + (loadCounter++) + "] for " + data + ": " + element);
                                            graph.asyncRequest(new LoadRequest(canvas, diagram, factory, clazz, data), loadListener);
                                        } else {
                                            //System.out.println("Spawn[" + (spawnCounter++) + "] for " + data + ": " + element);
                                            factory.load(graph, canvas, diagram, data, element, new AsyncProcedure<IElement>() {
                                                @Override
                                                public void exception(AsyncReadGraph graph, Throwable throwable) {
                                                    LOGGER.error("Unexpected error in ElementFactory.load (factory=" + factory + ")", throwable);
                                                }
                                                @Override
                                                public void execute(AsyncReadGraph graph, IElement result) {
                                                    // Loading complete, don't care.
                                                }
                                            });
                                        }
                                    }

                                });

                            }

                        });

//                        graph.asyncRequest(new SafeName(data), new Procedure<String>() {
//                            @Override
//                            public void exception(Throwable t) {
//                            }
//                            @Override
//                            public void execute(String result) {
//                                System.out.println("NodeRequest2 factory "  + result + " " + factory.getClass().getName());
//                            }
//                        });

                    }

                });

            }

        });

    }

    static int loadCounter = 0;
    static int spawnCounter = 0;

}
