/*******************************************************************************
 * Copyright (c) 2007, 2026 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *     Semantum Oy
 *******************************************************************************/
package org.simantics.diagram.adapter;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import org.simantics.databoard.Bindings;
import org.simantics.db.ReadGraph;
import org.simantics.db.RequestProcessor;
import org.simantics.db.Resource;
import org.simantics.db.common.primitiverequest.OrderedSet;
import org.simantics.db.common.procedure.wrapper.SetListenerToSingleSetListener;
import org.simantics.db.common.request.BinaryRead;
import org.simantics.db.common.utils.InvalidOrderedSetException;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.procedure.SetListener;
import org.simantics.diagram.stubs.DiagramResource;
import org.simantics.layer0.Layer0;
import org.simantics.modeling.ModelingResources;
import org.simantics.scenegraph.profile.Group;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * @author Tuukka Lehtonen
 */
public class MappedTypeGroup implements Group {

    private static final Logger LOGGER = LoggerFactory.getLogger(TypeGroup.class);

    private final List<Resource> types;
    private final String         name;

    public MappedTypeGroup(String name, Resource type) {
        this(name, new Resource[] { type });
    }

    public MappedTypeGroup(String name, Resource... types) {
        this.types = Arrays.asList(types);
        this.name = name;
    }

    /**
     * Initialize TypeGroup from a DIAGRAM.Group instance.
     * 
     * @param graph
     * @param group
     * @throws DatabaseException
     */
    public MappedTypeGroup(ReadGraph graph, final Resource group) throws DatabaseException {
        Layer0 L0 = Layer0.getInstance(graph);
        DiagramResource DIA = DiagramResource.getInstance(graph);

        this.name = graph.getPossibleRelatedValue(group, L0.HasName, Bindings.STRING);
        this.types = new ArrayList<Resource>();
        this.types.addAll(graph.getObjects(group, DIA.TypeGroup_HasType));
        
    }

    @Override
    public void trackItems(RequestProcessor processor, final Resource runtimeDiagram, final SetListener<Resource> listener) throws DatabaseException {
        if (types.isEmpty()) {
            System.out.println("MappedTypeGroup has no types!");
            return;
        }

        processor.syncRequest(new BinaryRead<Resource, Collection<Resource>, Collection<Resource>>(runtimeDiagram, types) {

            @Override
            public Set<Resource> perform(ReadGraph graph) throws DatabaseException {
                HashSet<Resource> result = new HashSet<Resource>();

                Resource realDiagram = graph.getPossibleObject(runtimeDiagram, DiagramResource.getInstance(graph).RuntimeDiagram_HasConfiguration);
                if (realDiagram == null)
                    return result;

                ModelingResources MOD = ModelingResources.getInstance(graph);

//                System.out.println("looking for diagram elements of type:");
//                for (Resource t : types)
//                    System.out.println("\t" + NameUtils.getSafeName(graph, t, true));

                if (graph.hasStatement(realDiagram)) {
                    try {
                        Collection<Resource> elements = graph.syncRequest(new OrderedSet(realDiagram));
                        for (Resource element : elements) {
//                            System.out.println("checking element " + NameUtils.getSafeName(graph, element, true));
                            Resource mapped = graph.getPossibleObject(element, MOD.ElementToComponent);
//                            System.out.println("checking mapped component " + NameUtils.getSafeName(graph, mapped, true));
                            if(mapped != null) {
                                Collection<Resource> mappedTypes = graph.getTypes(mapped);
//                                for (Resource t : mappedTypes)
//                                    System.out.println("\t" + NameUtils.getSafeName(graph, t, true));
                                if (!Collections.disjoint(types, mappedTypes))
                                    result.add(element);
                            } else {
                                Resource referenceRelation = graph.getPossibleObject(element, MOD.HasReferenceRelation);
                                if (referenceRelation != null) {
                                    Collection<Resource> referencedTypes = graph.getObjects(referenceRelation, MOD.HasReferencedType);
                                    if (!Collections.disjoint(types, referencedTypes)) {
                                        result.add(element);
                                    } else {
                                        for (Resource referencedType : referencedTypes) {
                                            Set<Resource> allTypes = graph.getSupertypes(referencedType);
                                            if (!Collections.disjoint(types, allTypes))
                                                result.add(element);
                                        }
                                    }
                                }
                            }
                        }
                    } catch (InvalidOrderedSetException e) {
                        LOGGER.debug("Most likely a diagram has been deleted and no ordered set can be found for {}", realDiagram);
                    }
                } else {
                    LOGGER.debug("Most likely a diagram has been deleted and no ordered set can be found for {}", realDiagram);
                }

                return result;
            }

        }, new SetListenerToSingleSetListener<Resource>(listener));
    }

    @Override
    public int hashCode() {
        return types.hashCode();
    }

    @Override
    public boolean equals(Object object) {
        if (this == object)
            return true;
        else if (object == null)
            return false;
        else if (MappedTypeGroup.class != object.getClass())
            return false;
        MappedTypeGroup other = (MappedTypeGroup)object;
        return types.equals(other.types);
    }

    @Override
    public String toString() {
        return "every '" + name + "'";
    }

}
