/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.diagram.adapter;

import org.simantics.db.ReadGraph;
import org.simantics.db.RequestProcessor;
import org.simantics.db.Resource;
import org.simantics.db.common.request.ReadRequest;
import org.simantics.db.exception.DatabaseException;
import org.simantics.diagram.stubs.DiagramResource;
import org.simantics.scenegraph.profile.EvaluationContext;
import org.simantics.scenegraph.profile.Group;
import org.simantics.scenegraph.profile.ProfileEntry;
import org.simantics.scenegraph.profile.Style;
import org.simantics.utils.ui.ErrorLogger;

public class GroupStyleProfileEntry implements ProfileEntry {

    private final Resource entry;
    private boolean        active = false;

    private Style          style;
    private Group          group;
    private double         priority;

    public GroupStyleProfileEntry(ReadGraph graph, Resource entry) throws DatabaseException {
        this.entry = entry;

        DiagramResource dr = DiagramResource.getInstance(graph);
        Resource style = graph.getPossibleObject(entry, dr.ProfileEntry_HasStyle);
        Resource group = graph.getPossibleObject(entry, dr.ProfileEntry_HasGroup);

        Double priority = graph.getPossibleRelatedValue(entry, dr.ProfileEntry_HasPriority);
        if(priority != null) this.priority = priority;
        else this.priority = 0;

        GroupStyleProfileEntry.this.style = graph.adapt(style, Style.class);
        GroupStyleProfileEntry.this.style.setPriority(this.priority);
        GroupStyleProfileEntry.this.group = graph.adapt(group, Group.class);
    }

    @Override
    public void activate(RequestProcessor processor, final Resource runtimeDiagram, final EvaluationContext observer) {

        try {
            processor.syncRequest(new ReadRequest() {

                @Override
                public void run(ReadGraph graph) throws DatabaseException {

                    GroupStyleProfileEntry.this.style.activate(graph, runtimeDiagram, entry, GroupStyleProfileEntry.this.group, observer);

                }

            });

            active = true;

        } catch (DatabaseException e) {
            ErrorLogger.defaultLogError(e);
        }
    }

    @Override
    public void deactivate(Resource diagram, EvaluationContext observer) {
        if (style == null) {
            System.out.println("GroupStyleProfileEntry has null style!");
            return;
        }

        style.deactivate(diagram, entry, group, observer);
        active = false;
    }

    @Override
    public void apply(EvaluationContext observer) {
        if (style == null) {
            System.out.println("GroupStyleProfileEntry has null style!");
            return;
        }
        style.apply(entry, group, observer);
    }

    @Override
    public int hashCode() {
        return entry.hashCode();
    }

    @Override
    public boolean equals(Object object) {
        if (this == object)
            return true;
        else if (object == null)
            return false;
        else if (GroupStyleProfileEntry.class != object.getClass())
            return false;
        GroupStyleProfileEntry other = (GroupStyleProfileEntry)object;
        return entry.equals(other.entry);
    }
    
    @Override
    public String toString() {
        return "" + style + " applied to " + group;
    }

    @Override
    public boolean isActive() {
        return active;
    }
    
    @Override
    public double getPriority() {
    	return priority;
    }

}
