/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.diagram.synchronization.graph;

import java.util.HashMap;

import org.simantics.databoard.Bindings;
import org.simantics.db.Resource;
import org.simantics.db.WriteGraph;
import org.simantics.db.common.CommentMetadata;
import org.simantics.db.common.request.IndexRoot;
import org.simantics.db.common.utils.OrderedSetUtils;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.adapter.Remover;
import org.simantics.db.layer0.exception.CannotRemoveException;
import org.simantics.db.layer0.util.RemoverUtil;
import org.simantics.diagram.content.ConnectionUtil;
import org.simantics.diagram.stubs.DiagramResource;
import org.simantics.diagram.synchronization.ModificationAdapter;
import org.simantics.layer0.Layer0;
import org.simantics.scl.commands.Commands;

/**
 * @author Tuukka Lehtonen
 */
public class RemoveElement extends ModificationAdapter {

    Resource diagram;
    Resource removedElement;

    public RemoveElement(Resource diagram, Resource removed) {
        super(REMOVE_NODE_PRIORITY);

        if(diagram == null)
            throw new NullPointerException();
        if(removed == null)
            throw new NullPointerException();
        
        this.diagram = diagram;
        this.removedElement = removed;
    }

    @Override
    public void perform(WriteGraph g) throws DatabaseException {
        Commands.get(g, "Simantics/Diagram/removeElement")
                .execute(g, g.syncRequest(new IndexRoot(diagram)), diagram, removedElement);
    }
    
    public static void removeElement(WriteGraph g, Resource diagram, Resource removedElement) throws DatabaseException {
        boolean connection = g.isInstanceOf(removedElement, DiagramResource.getInstance(g).Connection);
        String elementName = "";
        // Remove element from all layers if possible.
        DiagramResource DIA = DiagramResource.getInstance(g);
        g.deny(removedElement, DIA.IsVisible);
        g.deny(removedElement, DIA.IsFocusable);

        if (connection) {
            ConnectionUtil cu = new ConnectionUtil(g);
            cu.removeConnection(removedElement);
        } else {
            Remover r = RemoverUtil.getPossibleRemover(g, removedElement);
            if (r != null) {
                String problem = r.canRemove(g, new HashMap<Object, Object>(4));
                if (problem != null) {
                    throw new CannotRemoveException(problem);
                }
            }

            // Perform custom removals through ElementWriter adaption.
            // FIXME: should be able to use getSingleType if everything else goes correctly??
            Resource elementType = g.getPossibleType(removedElement, DiagramResource.getInstance(g).Element);
            if (elementType != null) {
                ElementWriter writer = g.adapt(elementType, ElementWriter.class);
                writer.removeFromGraph(g, removedElement);
            }

            // Remove element from diagram
            OrderedSetUtils.remove(g, diagram, removedElement);
            elementName = g.getPossibleRelatedValue2(removedElement, Layer0.getInstance(g).HasName, Bindings.STRING);
            RemoverUtil.remove(g, removedElement);
        }

        // Add comment to change set.
        CommentMetadata cm = g.getMetadata(CommentMetadata.class);
        g.addMetadata(cm.add("Removed element " + elementName + " " + removedElement));
    }
}
