package org.simantics.diagram.synchronization;

import org.simantics.db.Resource;
import org.simantics.db.WriteGraph;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.function.DbFunction;

/**
 * Interface for performing additional changes after copying a diagram node.
 * 
 * @author Reino Ruusu
 * @since 1.48.0
 */
public interface CopyFinisher {

	/**
	 * Make an update to a copy of a diagram node after the CopyAdvisor has created
	 * the copy.
	 * 
	 * <p>
	 * Default implementation does nothing but this allows implementers to only
	 * implement {@link #finishCopyWithPostAction(WriteGraph, Resource, Resource)} when
	 * necessary.
	 * </p>
	 * 
	 * @param graph  A write transaction
	 * @param source A source diagram node
	 * @param copy   A copy of the source diagram node, created within the ongoing
	 *               transaction
	 * @throws DatabaseException
	 * 
	 * @since 1.48.0
	 */
	default void finishCopy(WriteGraph graph, Resource source, Resource copy) throws DatabaseException {}

	/**
	 * A post-copy action that is executed at the end of an otherwise completed
	 * in-diagram copy operation.
	 * 
	 * <p>
	 * The action can return another <code>PostCopyAction</code> if for some reason
	 * it cannot yet perform its tasks but implementations must eventually return
	 * <code>null</code> to allow the copy process to finish if the action can't do
	 * anything more.
	 * </p>
	 * 
	 * @since 1.51.0
	 */
	public interface PostCopyAction extends DbFunction<WriteGraph, PostCopyAction> {}

	/**
	 * Make an update to a copy of a diagram node after the CopyAdvisor has created
	 * the copy.
	 * 
	 * <p>
	 * If the finisher can't yet complete its chores at the moment of invocation, it
	 * is allowed to return a {@link PostCopyAction} that can handle the rest of the
	 * chores after the copy operation has been otherwise completed.
	 * </p>
	 * 
	 * <p>
	 * The default implementation will simply invoke
	 * {@link #finishCopy(WriteGraph, Resource, Resource)} and return
	 * <code>null</code> for backwards compatibility.
	 * </p>
	 * 
	 * @param graph  A write transaction
	 * @param source A source diagram node
	 * @param copy   A copy of the source diagram node, created within the ongoing
	 *               transaction
	 * @return an optional continuation to this finish action to be executed as a
	 *         post-copy step before
	 *         {@link CopyAdvisor#onFinish(ISynchronizationContext)} is invoked.
	 *         Return <code>null</code> if not needed.
	 * @throws DatabaseException
	 * 
	 * @since 1.51.0
	 */
	default PostCopyAction finishCopyWithPostAction(WriteGraph graph, Resource source, Resource copy) throws DatabaseException {
		finishCopy(graph, source, copy);
		return null;
	}

}
