/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.diagram.handler;

import java.awt.geom.Point2D;
import java.lang.ref.WeakReference;
import java.util.Collections;
import java.util.Set;

import org.simantics.db.Resource;
import org.simantics.g2d.canvas.ICanvasContext;
import org.simantics.g2d.element.IElement;
import org.simantics.utils.strings.EString;

/**
 * @author Tuukka Lehtonen
 */
public class DiagramSelection {

    public static final DiagramSelection EMPTY = new DiagramSelection((Resource) null);

    // Reference may be null
    private final WeakReference<ICanvasContext> sourceCanvas;
    private final Resource source;

    /**
     * May be empty or at least invalid after the source canvas no longer
     * exists. Should not be used for anything besides visualisation.
     */
    private transient final Set<IElement> elements;

    private final ElementObjectAssortment assortment;
    private final boolean cut;
    private final Point2D copyPos;

    private DiagramSelection(Resource source) {
        this.sourceCanvas = null;
        this.source = source;
        this.elements = Collections.emptySet();
        this.assortment = ElementObjectAssortment.fromElements(elements);
        this.cut = false;
        this.copyPos = new Point2D.Double();
    }

    public DiagramSelection(ICanvasContext sourceCanvas, Resource source, Set<IElement> elements, boolean cut, Point2D copyPos) {
        assert source != null;
        assert elements != null;

        this.sourceCanvas = new WeakReference<ICanvasContext>(sourceCanvas);
        this.source = source;
        this.elements = Collections.unmodifiableSet(elements);
        this.assortment = ElementObjectAssortment.fromElements(elements);
        this.cut = cut;
        this.copyPos = copyPos;
    }

    /**
     * @return <code>null</code> if the source canvas has already been disposed.
     */
    public ICanvasContext getSourceCanvas() {
        ICanvasContext ctx = sourceCanvas == null ? null : sourceCanvas.get();
        return ctx == null ? null : ctx.isDisposed() ? null : ctx;
    }

    public boolean isFromCanvasContext(ICanvasContext ctx) {
        ICanvasContext srcCtx = getSourceCanvas();
        return ctx != null && ctx == srcCtx;
    }

    public Resource getSourceDiagram() {
        return source;
    }

    public boolean isCut() {
        return cut;
    }

    public Point2D getCopyPos() {
        return copyPos;
    }

    public Set<IElement> getOriginalElements() {
        return elements;
    }

    public ElementObjectAssortment getAssortment() {
        return assortment;
    }

    public boolean isEmpty() {
        return getAssortment().isEmpty();
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append("Diagram selection of ");
        sb.append(elements.size());
        sb.append(" element(s) ");
        if (cut)
            sb.append("cut");
        else
            sb.append("copied");
        sb.append(" from ");
        sb.append(source);
        sb.append(" with reference position ");
        sb.append(copyPos);
        if (elements != null && !elements.isEmpty()) {
            sb.append(":\n");
            String elems = EString.implode(elements, "\n");
            if (elems != null)
                sb.append(EString.addPrefix(elems, "\t"));
        }
        if (assortment != null && !assortment.isEmpty()) {
            sb.append(assortment.toString());
        }
        return sb.toString();
    }
}

