package org.simantics.diagram.function;

import java.util.Collection;
import java.util.HashSet;
import java.util.Set;

import org.simantics.Simantics;
import org.simantics.annotation.ontology.AnnotationResource;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.variable.Variable;
import org.simantics.db.layer0.variable.Variables;
import org.simantics.diagram.content.ConnectionUtil;
import org.simantics.diagram.flag.FlagUtil;
import org.simantics.diagram.stubs.DiagramResource;
import org.simantics.layer0.Layer0;
import org.simantics.modeling.ModelingResources;
import org.simantics.modeling.template2d.ontology.Template2dResource;
import org.simantics.structural.stubs.StructuralResource2;

public class PredefinedVariables {
	public static final String root = "root";
	public static final String project = "project";
	public static final String model = "model";
	public static final String template = "template";
	public static final String current = ".";
	public static final String diagramComposite = "diagramComposite";
	public static final String diagram = "diagram";
	public static final String flagAnnotation = "annotation";
	
	/** 
	 *  flagSourceObject	refers to a structural component which has an element on the flag's diagram  
	 */
	public static final String flagSourceObject = "flagSourceObject";

	/** 
	 *  flagTargetObject	refers to a structural component which has an element on the counterpart flag's diagram  
	 */
	public static final String flagTargetObject = "flagTargetObject";

	/** 
	 *  flagSourceComposite	refers to a structural composite which corresponds the flag's diagram  
	 */
	public static final String flagSourceComposite = "flagSourceComposite";

	/** 
	 *  flagTargetComposite	refers to a structural composite which corresponds the counterpart flag's diagram  
	 */
	public static final String flagTargetComposite = "flagTargetComposite";

	/** 
	 *  flagRouterComponent	refers to a structural component which corresponds a graphical connection connected with the flag  
	 */
	public static final String flagRouterComponent = "flagRouterComponent";

	private static PredefinedVariables factory = new PredefinedVariables();
	
	public static PredefinedVariables getInstance(){
		return factory;
	}
	
	public static void setFactory(PredefinedVariables factory){
		PredefinedVariables.factory = factory;
	}
	
    Resource connectedComponent = null;
    
    private Resource getConnectedComponent(ReadGraph graph, Resource flag) throws DatabaseException {
		StructuralResource2 STR = StructuralResource2.getInstance(graph);
		ModelingResources MOD = ModelingResources.getInstance(graph);
		ConnectionUtil cu = new ConnectionUtil(graph);
		Resource connectedComponent = null;
		Set<Resource> connectors = new HashSet<Resource>();
		for (Resource connector : graph.getObjects(flag, STR.IsConnectedTo)) {
		    Resource diagramConnection = ConnectionUtil.tryGetConnection(graph, connector);
		    if (diagramConnection != null) {
		        connectors.clear();
		        cu.getConnectors(diagramConnection, connectors);
		        connectors.remove(connector);
		        if (connectors.isEmpty()) {
		            continue;
		        } else {
		            //connectedComponent = graph.getPossibleObject(diagramConnection, MOD.ElementToComponent);
		            if (connectedComponent == null) {
		                for (Resource conn : connectors) {
		                    Resource element = cu.getConnectedComponent(diagramConnection, conn);
		                    if (element == null)
		                        continue;
		                    connectedComponent = graph.getPossibleObject(element, MOD.ElementToComponent);
		                    if (connectedComponent != null)
		                        break;
		                }
		            }
		        }
		    }
		}
		return connectedComponent;
    }

    public Resource getFlagRouterComponent(ReadGraph graph, Variable variable) throws DatabaseException {
        Resource flag = variable.getPossibleRepresents(graph);
        return flag == null ? null : getFlagRouterComponent(graph, flag);
    }

    public Resource getFlagRouterComponent(ReadGraph graph, Resource flag) throws DatabaseException {

        Layer0 L0 = Layer0.getInstance(graph);
        ModelingResources MOD = ModelingResources.getInstance(graph);

        Resource container = graph.getSingleObject(flag, L0.PartOf);
        Resource composite = graph.getSingleObject(container, MOD.DiagramToComposite);
        Variable var = Variables.getVariable(graph, composite);

//        System.err.println("var: " + var.getURI(graph));

        Variable result = FlagUtil.getPossibleFlagSignal(graph, var, flag, L0.Entity);
        if(result != null) return result.getRepresents(graph);
        else return null;

    }

	private Resource getFlagTargetObject(ReadGraph graph, Variable variable){
		try {
			Resource flag = variable.getRepresents(graph);
			Set<Resource> targetFlags = FlagUtil.getCounterparts(graph, flag);

			// if there are multiple targets
			if (targetFlags.size() != 1)
				return null;

			Resource targetFlag = targetFlags.toArray(new Resource[0])[0];
			if (targetFlag == null)
				return null;
	        Resource connectedComponent = getConnectedComponent(graph, targetFlag);
			if (connectedComponent == null)
				return null;
			return connectedComponent;
		} catch (DatabaseException e) {
		}
		return null;
	}

	private Resource getFlagSourceObject(ReadGraph graph, Variable variable){
		try {
			Resource flag = variable.getRepresents(graph);
			Resource connectedComponent = getConnectedComponent(graph, flag);
			if (connectedComponent == null)
				return null;
			return connectedComponent;
		} catch (DatabaseException e) {
		}
		return null;
	}
	
	private Resource getFlagSourceComposite(ReadGraph graph, Variable variable){
		try {
			Layer0 L0 = Layer0.getInstance(graph);
			ModelingResources MOD = ModelingResources.getInstance(graph);
			
			Resource flag = variable.getRepresents(graph);
			Resource diagram = graph.getPossibleObject(flag, L0.PartOf);
			if (diagram == null)
				return null;
			Resource composite = graph.getPossibleObject(diagram, MOD.DiagramToComposite);
			if (composite == null)
				return null;
			return composite;
		} catch (DatabaseException e) {
		}
		return null;
	}
	
	private Resource getFlagTargetComposite(ReadGraph graph, Variable variable){
		try {
			Layer0 L0 = Layer0.getInstance(graph);
			ModelingResources MOD = ModelingResources.getInstance(graph);
			
			Resource flag = variable.getRepresents(graph);
			Set<Resource> targetFlags = FlagUtil.getCounterparts(graph, flag);
			
			// if there are multiple targets
			if (targetFlags.size() != 1)
				return null;
			
			Resource targetFlag = targetFlags.toArray(new Resource[0])[0];
			if (targetFlag == null)
				return null;
			Resource diagram = graph.getPossibleObject(targetFlag, L0.PartOf);
			if (diagram == null)
				return null;
			Resource composite = graph.getPossibleObject(diagram, MOD.DiagramToComposite);
			if (composite == null)
				return null;
			return composite;
		} catch (DatabaseException e) {
		}
		return null;
	}

	public static Resource getAnnotation(ReadGraph graph, Resource res) throws DatabaseException {
		Layer0 L0 = Layer0.getInstance(graph);
        Collection<Resource> children = graph.getObjects(res, L0.ConsistsOf);
        Resource anno = null;
        AnnotationResource ANNO = AnnotationResource.getInstance(graph);
        for (Resource child:children){
        	if (graph.isInstanceOf(child, ANNO.Annotation)){
        		anno = child;
        		break;
        	}
        }
        return anno;
	} 

	private Resource getDiagramComposite(ReadGraph graph, Variable variable){
		try {
			Resource res = variable.getRepresents(graph);
			if (res == null)
				return null;
			DiagramResource DIA = DiagramResource.getInstance(graph);
			if (graph.isInstanceOf(res, DIA.Flag)){
				ModelingResources MOD = ModelingResources.getInstance(graph);
				Layer0 L0 = Layer0.getInstance(graph);
				Resource parent = graph.getPossibleObject(res, L0.PartOf);
				if (parent == null)
					return null;
				if (!graph.isInstanceOf(parent, DIA.Diagram))
					return null;
				return graph.getPossibleObject(parent, MOD.DiagramToComposite);
			}
			return res;
		} catch (DatabaseException e) {
		}
		return null;
	}

	public Resource getPredefinedResource(ReadGraph graph, Variable selection, String id) throws DatabaseException {
		Resource predefined = null;
        if (id.equals(root))
        	predefined = graph.getRootLibrary();
        else if (id.equals(project))
        	predefined = Simantics.getProject().get();
        else if (id.equals(model))
        	predefined = Variables.getPossibleModel(graph, selection);
        else if (id.equals(template)){
			Resource composite = selection.getRepresents(graph);
			if (composite == null)
				return null;
			ModelingResources MOD = ModelingResources.getInstance(graph);
            Template2dResource TEPLATE2D = Template2dResource.getInstance(graph);
			Resource diagram = graph.getPossibleObject(composite, MOD.CompositeToDiagram);
			if (diagram == null)
				return null;
			predefined = graph.getPossibleObject(diagram, TEPLATE2D.HasDrawingTemplate);
        }
        else if (id.equals(current))
        	predefined =  selection.getPossibleRepresents(graph);
        else if (id.equals(flagAnnotation)){
        	Resource flag = selection.getPossibleRepresents(graph);
        	if (flag != null)
        		predefined = getAnnotation(graph, flag);
        }
        else if (id.equals(diagram)){
			Resource composite = selection.getPossibleRepresents(graph);
			if (composite == null)
				return null;
			ModelingResources MOD = ModelingResources.getInstance(graph);
			predefined = graph.getPossibleObject(composite, MOD.CompositeToDiagram);
        }
        else if (id.equals(diagramComposite))
        	predefined = getDiagramComposite(graph, selection);
        else if (id.equals(flagSourceObject))
        	predefined = getFlagSourceObject(graph, selection);
        else if (id.equals(flagTargetObject))
        	predefined = getFlagTargetObject(graph, selection);
        else if (id.equals(flagSourceComposite))
        	predefined = getFlagSourceComposite(graph, selection);
        else if (id.equals(flagTargetComposite))
        	predefined = getFlagTargetComposite(graph, selection);
        else if (id.equals(flagRouterComponent))
        	predefined = getFlagRouterComponent(graph, selection);
        return predefined;
	}
	
	public Variable getPredefinedVariable(ReadGraph graph, Variable selection, String id) throws DatabaseException {
		Resource predefined = this.getPredefinedResource(graph, selection, id);
		if (predefined == null)
			return null;
		
        Variable v = selection;
        if (predefined != null)
            v = Variables.getPossibleVariable(graph, predefined);
        return v;
	}
	
	public Variable getVariable(ReadGraph graph, String path, Resource converter, Variable selection) throws DatabaseException {
        if (path == null)
        	return null;
        int colonInx = path.indexOf(':');
        int firstSlash = path.indexOf('/');
        int firstHash = path.indexOf('#');
        int firstFlashInx = firstSlash;
        if(firstFlashInx == -1) firstFlashInx = firstHash;
        else {
        	if(firstHash > -1 && firstHash < firstFlashInx) firstFlashInx = firstHash;
        }

        String predefined = null;
        String relativePath = null;

        // if scheme is followed by absolute uri
        // for examples:
        // pre:/model/Library/logo.svg#data -- get under model
        // pre:/./Library/logo.svg#data-- get under composite which corresponds the diagram
        // pre:/project/Library/FORTUM_LOGO_CMYK.svg#data -- get under project
        // pre:/root/Library/FORTUM_LOGO_CMYK.svg#data -- get under root
        
        if (colonInx != -1 && firstFlashInx != -1 && colonInx+1 == firstFlashInx && path.length() > firstFlashInx+1){
            String scheme = path.substring(0, colonInx);
            String absPath = path.substring(firstFlashInx+1);
            if (scheme.equals("pre")){
                int endOfPredefined1 = absPath.indexOf('/');
                int endOfPredefined2 = absPath.indexOf('#');
                if (endOfPredefined1 == -1 && endOfPredefined2 == -1)
                    predefined = absPath;
                if (endOfPredefined1 == -1 && endOfPredefined2 != -1)
                    predefined = absPath.substring(0, endOfPredefined2);
                if (endOfPredefined1 != -1 && endOfPredefined2 == -1)
                    predefined = absPath.substring(0, endOfPredefined1);
                if (endOfPredefined1 != -1 && endOfPredefined2 != -1){
                    if (endOfPredefined2 < endOfPredefined1)
                        endOfPredefined1 = endOfPredefined2;
                    predefined = absPath.substring(0, endOfPredefined1);
                }
                relativePath = absPath.substring(predefined.length());
            }
        }

        //Variable activeVariable = ScenegraphLoaderUtils.getPossibleVariableSelection(graph, context);
        if (selection == null)
        	return null;
        
        Variable v = selection;
        if (predefined != null)
        	v = getPredefinedVariable(graph, selection, predefined);

        if (v == null)
            return null;

        Variable property = null;
        if (relativePath != null){
        	if (relativePath.startsWith("/."))
        		relativePath = relativePath.substring(1);
            property = v.browsePossible(graph, relativePath);
        }
        else
            property = v.browsePossible(graph, path);
        return property;
    }

}
