/*******************************************************************************
 *  Copyright (c) 2007, 2024 Association for Decentralized Information Management
 *  in Industry THTH ry.
 *  All rights reserved. This program and the accompanying materials
 *  are made available under the terms of the Eclipse Public License v1.0
 *  which accompanies this distribution, and is available at
 *  http://www.eclipse.org/legal/epl-v10.html
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 *  Contributors:
 *      VTT Technical Research Centre of Finland - initial API and implementation
 *      Semantum Oy - improvements
 *******************************************************************************/
package org.simantics.diagram.elements;

import java.awt.Component;
import java.util.Collection;
import java.util.HashSet;
import java.util.Set;

import org.simantics.Simantics;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.WriteGraph;
import org.simantics.db.common.request.Queries;
import org.simantics.db.common.request.ResourceRead;
import org.simantics.db.common.request.WriteRequest;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.request.Read;
import org.simantics.db.service.TransactionSupport;
import org.simantics.db.service.VirtualGraphSupport;
import org.simantics.diagram.stubs.DiagramResource;
import org.simantics.diagram.ui.DiagramModelHints;
import org.simantics.g2d.canvas.ICanvasContext;
import org.simantics.g2d.chassis.AWTChassis;
import org.simantics.g2d.diagram.DiagramHints;
import org.simantics.g2d.diagram.IDiagram;
import org.simantics.g2d.diagram.handler.DataElementMap;
import org.simantics.g2d.element.ElementHints;
import org.simantics.g2d.element.IElement;
import org.simantics.g2d.layers.LayersConfiguration;
import org.simantics.g2d.scenegraph.ICanvasSceneGraphProvider;
import org.simantics.g2d.scenegraph.SceneGraphConfiguration;
import org.simantics.modeling.ModelingResources;
import org.simantics.scenegraph.INode;
import org.simantics.scenegraph.adapters.ISceneGraphProvider;
import org.simantics.scenegraph.g2d.G2DSceneGraph;
import org.simantics.scenegraph.g2d.IG2DNode;
import org.simantics.scenegraph.g2d.nodes.SingleElementNode;
import org.simantics.scenegraph.utils.NodeUtil;
import org.simantics.simulation.ontology.SimulationResource;

/**
 * Temporary transitory utils for handling some scene graph <-> g2d
 * interactions.
 * 
 * @author Antti Villberg
 */
public class DiagramNodeUtil {

    public static ICanvasContext getCanvasContext(IG2DNode node) {
        G2DSceneGraph sg = node.getRootNode2D();
        if (sg == null)
            return null;
        Component rootPane = sg.getRootPane();
        if(rootPane == null) {
        	throw new IllegalStateException("No root pane for scenegraph" + sg);
        }
        AWTChassis chassis = (AWTChassis)(rootPane.getParent());
        return chassis.getCanvasContext();
    }

    public static ICanvasContext getPossibleCanvasContext(IG2DNode node) {
        G2DSceneGraph sg = node.getRootNode2D();
        if (sg == null)
            return null;
        Component rootPane = sg.getRootPane();
        if(rootPane == null) {
        	return null;
        }
        AWTChassis chassis = (AWTChassis)(rootPane.getParent());
        return chassis.getCanvasContext();
    }

    public static IElement getElement(ICanvasContext ctx, IG2DNode node) {
        IDiagram diagram = ctx.getDefaultHintContext().getHint(DiagramHints.KEY_DIAGRAM);
        if (diagram != null) {
            SingleElementNode eNode = NodeUtil.findNearestParentNode(node, SingleElementNode.class);
            for (IElement e : diagram.getElements()) {
                INode n = e.getHint(ElementHints.KEY_SG_NODE);
                if (n != null && (node == n || eNode == n))
                    return e;
            }
        }
        return null;
    }

    public static IElement getElement(IG2DNode node) {
        ICanvasContext ctx = DiagramNodeUtil.getCanvasContext(node);
        if (ctx == null)
            return null;
        return getElement(ctx, node);
    }

    
    public static void activateProfileWithEntries(final Resource runtime, final Resource profile, final Collection<Resource> entries) throws DatabaseException {
        
        VirtualGraphSupport support = Simantics.getSession().getService(VirtualGraphSupport.class);
        Simantics.getSession().syncRequest(new WriteRequest(support.getWorkspacePersistent("profiles")) {

            @Override
            public void perform(WriteGraph graph) throws DatabaseException {
                DiagramResource DIA = DiagramResource.getInstance(graph);
                SimulationResource SIMU = SimulationResource.getInstance(graph);
                graph.deny(runtime, DIA.RuntimeDiagram_HasRuntimeProfile);
                graph.claim(runtime, DIA.RuntimeDiagram_HasRuntimeProfile, profile);
                for(Resource entry : entries)
                	graph.claim(profile, SIMU.IsActive, entry);
            }
            
        });
        
    }

    private static Read<Resource> diagramToComposite(Resource diagram) throws DatabaseException {
        return new ResourceRead<Resource>(diagram) {
            @Override
            public Resource perform(ReadGraph graph) throws DatabaseException {
                ModelingResources mr = ModelingResources.getInstance(graph);
                return graph.getPossibleObject(resource, mr.DiagramToComposite);
            }
        };
    }

    public static ICanvasSceneGraphProvider loadSceneGraphProvider(Resource model, Resource diagram, String diagramRVI) throws DatabaseException {
        Resource composite = Simantics.getSession().syncRequest( diagramToComposite(diagram) );
        return loadSceneGraphProviderForComposite(model, composite, diagramRVI);
    }

    public static ICanvasSceneGraphProvider loadSceneGraphProvider(ICanvasContext context, Resource model, Resource diagram, String diagramRVI) throws DatabaseException, InterruptedException {
    	// Default timeout is 30s
    	return loadSceneGraphProvider(context, model, diagram, diagramRVI, 30000);
    }

    public static ICanvasSceneGraphProvider loadSceneGraphProvider(ICanvasContext context, Resource model, Resource diagram, String diagramRVI, int timeoutMs) throws DatabaseException, InterruptedException {
        Resource composite = Simantics.getSession().syncRequest( diagramToComposite(diagram) );
        if(composite != null)
        	return loadSceneGraphProviderForComposite(context, model, composite, diagramRVI, timeoutMs);
        else
        	return loadSceneGraphProviderForDiagram(context, model, diagram, diagramRVI);
    }

    /**
     * @deprecated this leaks ICanvasSceneGraphProvider, use
     *             {@link #loadSceneGraphProvider(Resource, Resource, String)}
     *             instead
     */
    public static ICanvasContext loadDiagram(Resource model, Resource composite, String diagramRVI) throws DatabaseException, InterruptedException {
        String modelURI = Simantics.getSession().syncRequest( Queries.uri(model) );
        ISceneGraphProvider provider = Simantics.getSession().syncRequest( Queries.adapt( composite, ISceneGraphProvider.class ) );
        ICanvasSceneGraphProvider cp = (ICanvasSceneGraphProvider) provider;
        G2DSceneGraph sg = provider.initializeSceneGraph(new G2DSceneGraph(), modelURI, diagramRVI);

        NodeUtil.waitPending(cp.getCanvasContext().getThreadAccess(), sg);
        Simantics.getSession().getService(TransactionSupport.class).waitCompletion();

        return cp.getCanvasContext();
    }

    public static ICanvasSceneGraphProvider loadSceneGraphProviderForComposite(Resource model, Resource composite, String diagramRVI) throws DatabaseException {
        String modelURI = Simantics.getSession().syncRequest(Queries.uri(model));
        ISceneGraphProvider provider = Simantics.getSession().syncRequest( Queries.adapt( composite, ISceneGraphProvider.class ) );
        ICanvasSceneGraphProvider cp = (ICanvasSceneGraphProvider) provider;
        G2DSceneGraph sg = cp.initializeSceneGraph(new G2DSceneGraph(), modelURI, diagramRVI);

        NodeUtil.waitPending(cp.getCanvasContext().getThreadAccess(), sg);
        Simantics.getSession().getService(TransactionSupport.class).waitCompletion();

        return cp;
    }

    public static ICanvasSceneGraphProvider loadSceneGraphProviderForComposite(ICanvasContext context, Resource model, Resource composite, String diagramRVI) throws DatabaseException, InterruptedException {
    	// Default timeout is 30s
    	return loadSceneGraphProviderForComposite(context, model, composite, diagramRVI, 30000);
    }

    public static ICanvasSceneGraphProvider loadSceneGraphProviderForComposite(ICanvasContext context, Resource model, Resource composite, String diagramRVI, int timeoutMs) throws DatabaseException, InterruptedException {
        String modelURI = Simantics.getSession().syncRequest(Queries.uri(model));
        ISceneGraphProvider provider = Simantics.getSession().syncRequest( Queries.adapt( composite, ISceneGraphProvider.class ) );
        ICanvasSceneGraphProvider cp = (ICanvasSceneGraphProvider) provider;
        cp.initializeSceneGraph(context, modelURI, diagramRVI);
        NodeUtil.waitPending(cp.getCanvasContext().getThreadAccess(), context.getSceneGraph(), timeoutMs);
        Simantics.getSession().getService(TransactionSupport.class).waitCompletion();
        return cp;
    }

    public static ICanvasSceneGraphProvider loadSceneGraphProviderForDiagram(ICanvasContext context, Resource model, Resource diagram, String diagramRVI, LayersConfiguration layersConfiguration) throws DatabaseException, InterruptedException {
        String modelURI = Simantics.getSession().syncRequest(Queries.uri(model));
        ISceneGraphProvider provider = Simantics.getSession().syncRequest( Queries.adapt( diagram, ISceneGraphProvider.class ) );
        ICanvasSceneGraphProvider cp = (ICanvasSceneGraphProvider) provider;
        cp.initializeSceneGraph(context, new SceneGraphConfiguration(modelURI, diagramRVI, layersConfiguration));
        NodeUtil.waitPending(cp.getCanvasContext().getThreadAccess(), context.getSceneGraph());
        Simantics.getSession().getService(TransactionSupport.class).waitCompletion();
        return cp;
    }

    public static ICanvasSceneGraphProvider loadSceneGraphProviderForDiagram(ICanvasContext context, Resource model, Resource diagram, String diagramRVI) throws DatabaseException, InterruptedException {
        String modelURI = Simantics.getSession().syncRequest(Queries.uri(model));
        ISceneGraphProvider provider = Simantics.getSession().syncRequest( Queries.adapt( diagram, ISceneGraphProvider.class ) );
        ICanvasSceneGraphProvider cp = (ICanvasSceneGraphProvider) provider;
        cp.initializeSceneGraph(context, modelURI, diagramRVI);
        NodeUtil.waitPending(cp.getCanvasContext().getThreadAccess(), context.getSceneGraph());
        Simantics.getSession().getService(TransactionSupport.class).waitCompletion();
        return cp;
    }

    public static IElement findElement(ICanvasContext ctx, Resource element) {
        
        IDiagram diagram = ctx.getDefaultHintContext().getHint(DiagramHints.KEY_DIAGRAM);
        DataElementMap dmap = diagram.getDiagramClass().getSingleItem(DataElementMap.class);
        return dmap.getElement(diagram, element);
        
    }

    public static INode findNode(ICanvasContext ctx, Resource element) {
        
        IElement e = findElement(ctx, element);
        return e.getHint(ElementHints.KEY_SG_NODE);
        
    }

    public static Resource getRuntime(ICanvasContext ctx) {
        
        IDiagram diagram = ctx.getDefaultHintContext().getHint(DiagramHints.KEY_DIAGRAM);
        return diagram.getHint(DiagramModelHints.KEY_DIAGRAM_RUNTIME_RESOURCE);
        
    }

    public static Resource getElementResource(Resource configuration) throws DatabaseException {
        
        return Simantics.getSession().syncRequest(new ResourceRead<Resource>(configuration) {

            @Override
            public Resource perform(ReadGraph graph) throws DatabaseException {
                return graph.getPossibleObject(resource, ModelingResources.getInstance(graph).ComponentToElement);
            }
            
        });
        
    }
    
    public static final Set<String> getTexts(Set<TextNode> nodes) {
        Set<String> result = new HashSet<String>(nodes.size());
        for(TextNode n : nodes) result.add(n.getText());
        return result;
    }
    
}
