package org.simantics.diagram.adapter;

import java.awt.geom.AffineTransform;
import java.awt.geom.Point2D;
import java.util.ArrayList;

import org.simantics.Simantics;
import org.simantics.databoard.Bindings;
import org.simantics.db.AsyncReadGraph;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.WriteGraph;
import org.simantics.db.common.procedure.guarded.GuardedAsyncProcedureWrapper;
import org.simantics.db.common.request.WriteRequest;
import org.simantics.db.common.utils.NameUtils;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.procedure.AsyncProcedure;
import org.simantics.diagram.stubs.G2DResource;
import org.simantics.diagram.synchronization.graph.DiagramGraphUtil;
import org.simantics.g2d.canvas.ICanvasContext;
import org.simantics.g2d.diagram.IDiagram;
import org.simantics.g2d.element.ElementClass;
import org.simantics.g2d.element.ElementUtils;
import org.simantics.g2d.element.IElement;
import org.simantics.g2d.element.handler.impl.DefaultTransform;
import org.simantics.g2d.element.handler.impl.RequireImageFocusMode;
import org.simantics.g2d.element.handler.impl.SimpleElementLayers;
import org.simantics.g2d.element.handler.impl.StaticObjectAdapter;
import org.simantics.g2d.element.handler.impl.StaticSymbolImpl;
import org.simantics.g2d.image.DefaultImages;
import org.simantics.g2d.image.Image;
import org.simantics.g2d.svg.SVGImage;
import org.simantics.scenegraph.g2d.nodes.IPathListener;
import org.simantics.scenegraph.g2d.nodes.PathNodeStyle;


/**
 * Element(Class)Factory for PathElements
 */
public class PathElementClassFactory extends SyncElementFactory {

    public static final ElementFactory INSTANCE = new PathElementClassFactory();

    @Override
    public ElementClass create(ReadGraph graph, ICanvasContext canvas, IDiagram diagram, Resource elementType) throws DatabaseException {
        String id = "PathElement: " + NameUtils.getSafeName(graph, elementType);
        G2DResource g2d = G2DResource.getInstance(graph);
        String svgDoc = graph.getPossibleRelatedValue(elementType, g2d.HasSVGDocument);
        Image image = null;
        if (svgDoc != null)
            image = new SVGImage(id+".svg", svgDoc);
        else
            image = DefaultImages.ERROR_DECORATOR.get();

        return ElementClass.compile(
                new StaticObjectAdapter(elementType),
                DefaultTransform.INSTANCE,
                SimpleElementLayers.INSTANCE,
                RequireImageFocusMode.INSTANCE,
                new PathElementHandler(),
                new StaticSymbolImpl(image)
        )
        .setId(id);
    }

    @Override
    public void load(AsyncReadGraph graph, final ICanvasContext canvas, final IDiagram diagram,
            final Resource elementResource, final IElement element, final AsyncProcedure<IElement> procedure) {
        GuardedAsyncProcedureWrapper<IElement> guard = new GuardedAsyncProcedureWrapper<IElement>(procedure, 2);
        super.load(graph, canvas, diagram, elementResource, element, guard);
        ElementFactoryUtil.loadLayersForElement(graph, diagram, element, elementResource, guard);
    }

    @Override
    public void load(ReadGraph graph, ICanvasContext canvas, IDiagram diagram, Resource element, IElement e) throws DatabaseException {
        G2DResource G2D = G2DResource.getInstance(graph);

        // Load transform
        AffineTransform at = DiagramGraphUtil.getAffineTransform(graph, element);
        ElementUtils.setTransform(e, at);

        double coordinates[] = DiagramGraphUtil.getPossibleRelatedDoubleArray(graph, element, G2D.HasPoint2DArray);
        if (coordinates != null) {
            int count = coordinates.length / 2;
            ArrayList<Point2D> points = new ArrayList<>(count);
            for (int i = 0; i < count; i++) {
                points.add(new Point2D.Double(coordinates[i*2], coordinates[i*2+1]));
            }
            e.setHint(PathElementHandler.KEY_PATH_POINTS, points);
        }
        
        boolean closed = graph.getRelatedValue(element, G2D.IsClosedPath);
        e.setHint(PathElementHandler.KEY_PATH_CLOSED, closed);

        Resource color = graph.getSingleObject(element, G2D.HasColor);
        float[] colorF = graph.getValue(color, Bindings.FLOAT_ARRAY);
        Resource fill = graph.getSingleObject(element, G2D.HasFillColor);
        float[] fillF = graph.getValue(fill, Bindings.FLOAT_ARRAY);
        float strokeWidth = graph.getRelatedValue(element, G2D.HasStrokeWidth);
        e.setHint(PathElementHandler.KEY_PATH_STYLE, new PathNodeStyle(colorF, fillF, strokeWidth));

        e.setHint(PathElementHandler.KEY_PATH_LISTENER, new IPathListener() {

            @Override
            public void pathChanged(ArrayList<Point2D> points, boolean closed) {

                int count = points.size();
                double coordinates[] = new double[count * 2];
                int i = 0;
                for (Point2D point : points) {
                    coordinates[i++] = point.getX();
                    coordinates[i++] = point.getY();
                }
                Simantics.async(new WriteRequest() {

                    @Override
                    public void perform(WriteGraph graph) throws DatabaseException {
                        graph.markUndoPoint();
                        graph.claimLiteral(element, G2D.HasPoint2DArray, coordinates);
                        graph.claimLiteral(element, G2D.IsClosedPath, closed);
                    }
                    
                });

            }
        });
    }

}
