/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.diagram.adapter;

import org.simantics.db.AsyncReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.common.primitiverequest.Adapter;
import org.simantics.db.common.procedure.adapter.TransientCacheAsyncListener;
import org.simantics.db.procedure.AsyncListener;
import org.simantics.db.procedure.AsyncProcedure;
import org.simantics.db.procedure.Listener;
import org.simantics.diagram.synchronization.ErrorHandler;
import org.simantics.g2d.canvas.ICanvasContext;
import org.simantics.g2d.diagram.IDiagram;
import org.simantics.g2d.element.ElementClass;
import org.simantics.g2d.element.IElement;

/**
 * @author Antti Villberg
 */
public class ConnectionRequest extends BaseRequest2<Resource, IElement> {

    final IDiagram diagram;
    final Listener<IElement> loadListener;
    final ErrorHandler errorHandler;

    public ConnectionRequest(ICanvasContext canvas, IDiagram diagram, Resource resource, ErrorHandler errorHandler, Listener<IElement> loadListener) {
        super(canvas, resource);
        this.diagram = diagram;
        this.errorHandler = errorHandler;
        this.loadListener = loadListener;
    }

    private static class ConnectionRequestListener implements AsyncListener<IElement> {

        private Listener<IElement> loadListener;
        private AsyncProcedure<IElement> procedure;

        ConnectionRequestListener(Listener<IElement> loadListener, AsyncProcedure<IElement> procedure) {
            this.loadListener = loadListener;
            this.procedure = procedure;
        }

        @Override
        public void execute(AsyncReadGraph graph, IElement result) {
            loadListener.execute(result);
            if(procedure != null) {
                procedure.execute(graph, result);
                procedure = null;
            }
        }

        @Override
        public void exception(AsyncReadGraph graph, Throwable t) {
            loadListener.exception(t);
            if(procedure != null) {
                procedure.exception(graph, t);
                procedure = null;
            }
        }

        @Override
        public boolean isDisposed() {
            return loadListener.isDisposed();
        }

    }

    @Override
    public void perform(AsyncReadGraph graph, final AsyncProcedure<IElement> procedure) {

        graph.forHasStatement(data, new AsyncProcedure<Boolean>() {

            @Override
            public void exception(AsyncReadGraph graph, Throwable throwable) {
                procedure.exception(graph, throwable);
            }

            @Override
            public void execute(AsyncReadGraph graph, Boolean result) {

                if (!result) {
                    procedure.execute(graph, null);
                    return;
                }

                graph.asyncRequest(new Adapter<ElementFactory>(data, ElementFactory.class), new TransientCacheAsyncListener<ElementFactory>() {

                    @Override
                    public void exception(AsyncReadGraph graph, Throwable throwable) {
                        errorHandler.error("Unexpected ElementFactory adaption failure", throwable);
                        procedure.execute(graph, null);
                    }

                    @Override
                    public void execute(AsyncReadGraph graph, final ElementFactory factory) {

                        graph.asyncRequest(new GetElementClassRequest(factory, data, canvas, diagram), new TransientCacheAsyncListener<ElementClass>() {

                            @Override
                            public void exception(AsyncReadGraph graph, Throwable throwable) {
                                errorHandler.error("Unexpected ElementClass creation failure", throwable);
                                procedure.execute(graph, null);
                            }

                            @Override
                            public void execute(AsyncReadGraph graph, final ElementClass ec) {

                                if (loadListener != null) {

                                    graph.asyncRequest(new LoadRequest(canvas, diagram, factory, ec, data), new ConnectionRequestListener(loadListener, procedure));

                                } else {

                                    graph.asyncRequest(new SpawnRequest(canvas, ec, data), new TransientCacheAsyncListener<IElement>() {

                                        @Override
                                        public void exception(AsyncReadGraph graph, Throwable throwable) {
                                            errorHandler.error("Unexpected SpawnRequest failure", throwable);
                                            procedure.exception(graph, throwable);
                                        }

                                        @Override
                                        public void execute(AsyncReadGraph graph, IElement element) {

                                            procedure.execute(graph, element);

                                            factory.load(graph, canvas, diagram, data, element, new AsyncProcedure<IElement>() {
                                                @Override
                                                public void exception(AsyncReadGraph graph, Throwable throwable) {
                                                    errorHandler.error("Unexpected ElementFactory.load failure", throwable);
                                                }
                                                @Override
                                                public void execute(AsyncReadGraph graph, IElement result) {
                                                    // Loading complete, don't care.
                                                }
                                            });
                                        }
                                    });
                                }
                            }
                        });
                    }
                });
            }
        });
    }

}
