/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.diagram;

import java.awt.BasicStroke;
import java.awt.Color;
import java.awt.Font;
import java.awt.geom.AffineTransform;
import java.awt.geom.Rectangle2D;
import java.util.Set;

import org.simantics.databoard.Bindings;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.WriteGraph;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.exception.DoesNotContainValueException;
import org.simantics.db.exception.ManyObjectsForFunctionalRelationException;
import org.simantics.db.exception.NoSingleResultException;
import org.simantics.db.exception.ServiceException;
import org.simantics.diagram.stubs.DiagramResource;
import org.simantics.diagram.stubs.G2DResource;
import org.simantics.diagram.synchronization.graph.DiagramGraphUtil;
import org.simantics.g2d.svg.LineCap;
import org.simantics.g2d.svg.LineJoin;
import org.simantics.layer0.Layer0;
import org.simantics.utils.strings.format.MetricsFormat;


/**
 * Utility class for handling G2D ontology concepts.
 * 
 * @author Marko Luukkainen <marko.luukkainen@vtt.fi>
 */
public class G2DUtils {

    /**
     * Utility to wrap resource to a object
     * 
     * @param g
     * @param resource
     * @return
     * @throws ServiceException
     * @throws NoSingleResultException
     * @throws DoesNotContainValueException
     * @throws ManyObjectsForFunctionalRelationException
     */
    public static Object getObject(ReadGraph g, Resource resource) throws ServiceException, NoSingleResultException, DoesNotContainValueException, ManyObjectsForFunctionalRelationException {
        G2DResource g2d = G2DResource.getInstance(g);
        Set<Resource> types = g.getTypes(resource);
        if (types.contains(g2d.Font)) {
            return getFont(g, resource);
        }
        if (types.contains(g2d.Color)) {
            return getColor(g, resource);
        }
        if (types.contains(g2d.Transform)) {
            return getTransform(g, resource);
        }
        if (types.contains(g2d.Rectangle2D)) {
            return getRectangle(g, resource);
        }

        return g.getPossibleValue(resource);
    }

    public static Font getFont(ReadGraph g, Resource fontResource) throws ServiceException, NoSingleResultException, DoesNotContainValueException, ManyObjectsForFunctionalRelationException {
        G2DResource g2d = G2DResource.getInstance(g);
        if (!g.isInstanceOf(fontResource, g2d.Font))
            throw new IllegalArgumentException("Resource " + fontResource + " is not a font");

        String name = g.getRelatedValue(fontResource, g2d.HasFontFamily);
        int size = g.getRelatedValue(fontResource, g2d.HasFontSize);
        int style = Font.PLAIN;
        Resource styleResource = g.getPossibleObject(fontResource, g2d.HasFontStyle);
        if (styleResource != null) {
            if (styleResource.equals(g2d.FontStyle_bold_italic_style)) {
                style = Font.BOLD | Font.ITALIC;
            } else if (styleResource.equals(g2d.FontStyle_bold_font_style)) {
                style = Font.BOLD;
            } else if (styleResource.equals(g2d.FontStyle_italic_font_style)) {
                style = Font.ITALIC;
            } else if (styleResource.equals(g2d.FontStyle_normal_font_style)) {
                style = Font.PLAIN;
            } else {
                throw new IllegalArgumentException("Given font " + fontResource + " does not contain valid style. Current style is " + styleResource);
            }
        }

        return new Font(name, style, size);
    }

    public static Color getColor(ReadGraph g, Resource colorResource) throws DoesNotContainValueException, ServiceException {
        G2DResource g2d = G2DResource.getInstance(g);
        if (!g.isInstanceOf(colorResource, g2d.Color))
            throw new IllegalArgumentException("Resource " + colorResource + " is not a color");

        float value[] = (float[])g.getValue(colorResource);
        if (value.length != 4)
            throw new IllegalArgumentException("Color " + colorResource + " does not have proper definition, expected 4 components, got " + value.length + " components");

        return new Color(value[0], value[1], value[2], value[3]);
    }

    public static AffineTransform getTransform(ReadGraph g, Resource resource) throws DoesNotContainValueException, ServiceException {
        G2DResource g2d = G2DResource.getInstance(g);
        if (!g.isInstanceOf(resource, g2d.Transform))
            throw new IllegalArgumentException("Resource " + resource + " is not a transform");

        double values[] = (double[])g.getValue(resource);
        if (values.length != 6)
            throw new IllegalArgumentException("Transform " + resource + " does not have proper definition, expected 6 components, got " + values.length + " components");

        return new AffineTransform(values);
    }

    public static Rectangle2D getRectangle(ReadGraph g, Resource resource) throws DoesNotContainValueException, ServiceException {
        G2DResource g2d = G2DResource.getInstance(g);
        if (!g.isInstanceOf(resource, g2d.Rectangle2D))
            throw new IllegalArgumentException("Resource " + resource + " is not a rectangle");

        double values[] = (double[])g.getValue(resource);
        if (values.length != 4)
            throw new IllegalArgumentException("Transform " + resource + " does not have proper definition, expected 4 components, got " + values.length + " components");

        return new Rectangle2D.Double(values[0], values[1], values[2], values[3]);
    }

    public static Resource createFont(WriteGraph g, Font font) throws ServiceException, ManyObjectsForFunctionalRelationException {
    	Layer0 b = Layer0.getInstance(g);
        G2DResource g2d = G2DResource.getInstance(g);
        Resource fontResource = g.newResource();
        g.claim(fontResource, b.InstanceOf, null, g2d.Font);
        g.claimLiteral(fontResource, g2d.HasFontFamily, font.getFamily());
        g.claimLiteral(fontResource, g2d.HasFontSize, font.getSize());
        if (font.getStyle() == (Font.BOLD|Font.ITALIC)) {
            g.claim(fontResource, g2d.HasFontStyle, g2d.FontStyle_bold_italic_style);
        } else if (font.getStyle() == Font.BOLD) {
            g.claim(fontResource, g2d.HasFontStyle, g2d.FontStyle_bold_font_style);
        } else if (font.getStyle() == Font.ITALIC) {
            g.claim(fontResource, g2d.HasFontStyle, g2d.FontStyle_italic_font_style);
        } else {
            g.claim(fontResource, g2d.HasFontStyle, g2d.FontStyle_normal_font_style);
        }

        return fontResource;
    }

    public static Resource createColor(WriteGraph g, Color color) throws ServiceException {
    	Layer0 b = Layer0.getInstance(g);
        G2DResource g2d = G2DResource.getInstance(g);
        Resource colorResource = g.newResource();
        g.claim(colorResource, b.InstanceOf, null, g2d.Color);
        g.claimValue(colorResource, color.getRGBComponents(null));
        return colorResource;
    }

    /*
     * Metrics Format
     */

    public static Resource createMetricsFormat(WriteGraph g, MetricsFormat format) throws ServiceException, ManyObjectsForFunctionalRelationException {
    	Layer0 b = Layer0.getInstance(g);
        DiagramResource dr = DiagramResource.getInstance(g);
        Resource formatResource = g.newResource();
        g.claim(formatResource, b.InstanceOf, null, dr.Format);
        g.claimLiteral(formatResource, b.HasName, format.getName());
        g.claimLiteral(formatResource, dr.HasPattern, format.getPattern());
        return formatResource;
    }

    public static MetricsFormat getMetricsFormat(ReadGraph g, Resource formatResource) throws NoSingleResultException, DoesNotContainValueException, ServiceException {
        Layer0 l0 = Layer0.getInstance(g);
        DiagramResource dr = DiagramResource.getInstance(g);
        String name = g.getRelatedValue(formatResource, l0.HasName);
        String pattern = g.getRelatedValue(formatResource, dr.HasPattern);
        return new MetricsFormat(pattern,1.0,name);
    }

    public static final Float DEFAULT_DASH_OFFSET  = 0f;
    public static final Float DEFAULT_STROKE_WIDTH = 1f;
    public static final Float DEFAULT_MITER_LIMIT  = 10f;

    public static BasicStroke getStroke(ReadGraph graph, Resource stroke) throws DatabaseException {
        return getStroke(graph, stroke, DEFAULT_DASH_OFFSET, DEFAULT_STROKE_WIDTH, DEFAULT_MITER_LIMIT);
    }

    public static BasicStroke getStroke(ReadGraph graph, Resource stroke, Float defaultDashOffset, Float defaultStrokeWidth, Float defaultMiterLimit) throws DatabaseException {
        if (stroke == null)
            return null;

        G2DResource g2d = G2DResource.getInstance(graph);
        float[] dashArray = DiagramGraphUtil.getPossibleRelatedValue(graph, stroke, g2d.HasDashArray, float[].class, null);
        Float dashOffset = DiagramGraphUtil.getPossibleRelatedValue(graph, stroke, g2d.HasDashOffset, Float.class, defaultDashOffset);
        Float strokeWidth = DiagramGraphUtil.getPossibleRelatedValue(graph, stroke, g2d.HasStrokeWidth, Float.class, defaultStrokeWidth);
        Float miterLimit = DiagramGraphUtil.getPossibleRelatedValue(graph, stroke, g2d.HasMiterLimit, Float.class, defaultMiterLimit);
        LineJoin lineJoin = DiagramGraphUtil.toLineJoin(g2d, graph.getPossibleObject(stroke, g2d.HasLineJoin));
        LineCap lineCap = DiagramGraphUtil.toLineCap(g2d, graph.getPossibleObject(stroke, g2d.HasLineCap));

        // Sanity checks.
        if (strokeWidth < 0)
            strokeWidth = 0f;

        return new BasicStroke(strokeWidth, lineCap.ordinal(), lineJoin.ordinal(), miterLimit, dashArray, dashOffset);
    }

    public static Resource createStroke(WriteGraph graph, BasicStroke stroke) throws DatabaseException {
        if (stroke == null)
            return null;

        Layer0 L0 = Layer0.getInstance(graph);
        G2DResource g2d = G2DResource.getInstance(graph);
        Resource result = graph.newResource();
        graph.claim(result, L0.InstanceOf, null, g2d.Stroke);
        if (stroke.getDashArray() != null)
            graph.claimLiteral(result, g2d.HasDashArray, L0.FloatArray, stroke.getDashArray(), Bindings.FLOAT_ARRAY);
        graph.claimLiteral(result, g2d.HasDashOffset, L0.Float, stroke.getDashPhase(), Bindings.FLOAT);
        graph.claimLiteral(result, g2d.HasStrokeWidth, L0.Float, stroke.getLineWidth(), Bindings.FLOAT);
        graph.claimLiteral(result, g2d.HasMiterLimit, L0.Float, stroke.getMiterLimit(), Bindings.FLOAT);
        graph.claim(result, g2d.HasLineJoin, null, DiagramGraphUtil.toLineJoin(g2d, LineJoin.values()[stroke.getLineJoin()]));
        graph.claim(result, g2d.HasLineCap, null, DiagramGraphUtil.toLineCap(g2d, LineCap.values()[stroke.getEndCap()]));

        return result;
    }

}
