/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.diagram.synchronization;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import org.simantics.utils.datastructures.hints.IHintObservable;
import org.simantics.utils.datastructures.hints.IHintContext.Key;

/**
 * Synchronizer that is composed of several other synchronizers. It will consult
 * the composed synchronizers in the order in which they were specified. Should
 * a composed synchronizer report that it has handled the synchronization, i.e.
 * returns <code>true</code>, the consultation will end there.
 * 
 * @author Marko Luukkainen <marko.luukkainen@vtt.fi>
 * @author Tuukka Lehtonen
 */
public class CompositeHintSynchronizer implements IHintSynchronizer {

    private final List<IHintSynchronizer> synchronizers = new ArrayList<IHintSynchronizer>();

    public CompositeHintSynchronizer() {
    }

    public CompositeHintSynchronizer(IHintSynchronizer... synchronizers) {
        for (IHintSynchronizer s : synchronizers)
            this.synchronizers.add(s);
    }

    public CompositeHintSynchronizer(Collection<IHintSynchronizer> synchronizers) {
        this.synchronizers.addAll(synchronizers);
    }

    public CompositeHintSynchronizer add(IHintSynchronizer s) {
        synchronizers.add(s);
        return this;
    }

    @Override
    public int synchronize(ISynchronizationContext context, IHintObservable observable) {
        int count = 0;
        for (IHintSynchronizer synchronizer : synchronizers) {
            count += synchronizer.synchronize(context, observable);
        }
        return count;
    }

    @Override
    public boolean hintChanged(ISynchronizationContext context, IHintObservable sender, Key key, Object oldValue,
            Object newValue) {
        for (IHintSynchronizer s : synchronizers) {
            if (s.hintChanged(context, sender, key, oldValue, newValue))
                return true;
        }
        return false;
    }

    @Override
    public boolean hintRemoved(ISynchronizationContext context, IHintObservable sender, Key key, Object oldValue) {
        for (IHintSynchronizer s : synchronizers) {
            if (s.hintRemoved(context, sender, key, oldValue))
                return true;
        }
        return false;
    }

}
