package org.simantics.diagram.symbolcontribution;

import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.common.primitiverequest.PossibleAdapter;
import org.simantics.db.common.procedure.adapter.TransientCacheAsyncListener;
import org.simantics.db.common.utils.NameUtils;
import org.simantics.db.exception.DatabaseException;
import org.simantics.diagram.internal.Activator;
import org.simantics.diagram.stubs.DiagramResource;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * @author Tuukka Lehtonen
 */
public class SymbolProviderFactories {

    private static final Logger LOGGER = LoggerFactory.getLogger(SymbolProviderFactories.class);

    private static boolean testFilter(ReadGraph graph, Resource contribution, Resource filterR, Resource diagram) throws DatabaseException {
        boolean trace = LOGGER.isTraceEnabled();
        try {
            if (trace)
                LOGGER.trace("testFilter({}, {}, {})",
                        NameUtils.getURIOrSafeNameInternal(graph, contribution),
                        NameUtils.getURIOrSafeNameInternal(graph, filterR),
                        NameUtils.getURIOrSafeNameInternal(graph, diagram));
            var filter = graph.syncRequest(
                    new PossibleAdapter<SymbolContributionFilter>(filterR, SymbolContributionFilter.class),
                    TransientCacheAsyncListener.<SymbolContributionFilter> instance());
            if (trace)
                LOGGER.trace("\tfilter: {}", filter);
            if (filter == null || !filter.accept(graph, contribution, filterR, diagram)) {
                if (trace)
                    LOGGER.trace("\t\treject");
                return false;
            }
            if (trace)
                LOGGER.trace("\t\taccept");
            return true;
        } catch (DatabaseException e) {
            // If any errors happen when requesting for the filter, we can
            // only assume that the graph related to their definitions is
            // somehow corrupted. In this case we must assume that the filter
            // did not pass since it did not work properly and true checking
            // could not be performed to completion.
            Activator.getDefault().getLog().log(new Status(IStatus.ERROR, Activator.PLUGIN_ID, "Filter " + filterR + " testing failed", e));
            return false;
        }
    }

    public static boolean accept(ReadGraph graph, DiagramResource DIA, Resource contribution, Resource diagram) throws DatabaseException {
        String name = graph.getPossibleAdapter(contribution, String.class);
        if (name == null)
            return false;

        for (Resource filterR : graph.getObjects(contribution, DIA.HasSymbolContributionFilter)) {
            if (!testFilter(graph, contribution, filterR, diagram)) {
                return false;
            }
        }
        return true;
    }

}
