/*******************************************************************************
 * Copyright (c) 2024 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Semantum Oy - initial API and implementation
 *******************************************************************************/
package org.simantics.diagram.symbolcontribution;

import java.nio.ByteBuffer;
import java.nio.LongBuffer;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;

import org.simantics.db.Resource;

/**
 * @author Tuukka Lehtonen
 * @since 1.60.0
 */
final class Hasher {

	private final MessageDigest md;
	private final ByteBuffer buf = ByteBuffer.wrap(new byte[128]);
	private final LongBuffer lbuf = buf.asLongBuffer();

	public Hasher() {
		this("SHA-256");
	}

	public Hasher(String algorithm) {
		try {
			this.md = MessageDigest.getInstance(algorithm); //$NON-NLS-1$
		} catch (NoSuchAlgorithmException e) {
			throw new RuntimeException(algorithm + " digest not supported by JVM"); //$NON-NLS-1$
		}
	}

	public void flush() {
		int pos = buf.position();
		if (pos > 0) {
			buf.position(0).limit(pos);
			md.update(buf);
			buf.position(0).limit(buf.capacity());
			lbuf.position(0).limit(lbuf.capacity());
		}
	}

	public void update(Resource r) {
		if (lbuf.remaining() == 0) {
			flush();
		}
		lbuf.put(r.getResourceId());
		buf.position(buf.position() + 8);
	}

	public void update(Resource r, Resource r2) {
		if (lbuf.remaining() < 2) {
			flush();
		}
		lbuf.put(r.getResourceId());
		lbuf.put(r2.getResourceId());
		buf.position(buf.position() + 16);
	}

	public void update(Resource r, Resource r2, Resource r3) {
		if (lbuf.remaining() < 3) {
			flush();
		}
		lbuf.put(r.getResourceId());
		lbuf.put(r2.getResourceId());
		lbuf.put(r3.getResourceId());
		buf.position(buf.position() + 24);
	}

	public void update(Resource... rs) {
		for (var r : rs) {
			update(r);
		}
	}

	public byte[] digest() {
		flush();
		return md.digest();
	}

}