/*******************************************************************************
 * Copyright (c) 2013 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Semantum Oy - initial API and implementation
 *******************************************************************************/
package org.simantics.diagram.symbolcontribution;

import org.eclipse.core.runtime.IAdaptable;

/**
 * A basic triple structure that implements IAdaptable. It tries to adapt the
 * objects to the requested class in order (first, second, third).
 * 
 * @author Tuukka Lehtonen
 * 
 * @param <T1> type of first
 * @param <T2> type of second
 * @param <T3> type of third
 */
final class AdaptableTriple<T1, T2, T3> implements IAdaptable {
    public final T1 first;
    public final T2 second;
    public final T3 third;
    private final int hash;

    public static <T1, T2, T3> AdaptableTriple<T1, T2, T3> make(T1 t1, T2 t2, T3 t3) {
        return new AdaptableTriple<T1, T2, T3>(t1, t2, t3);
    }

    public AdaptableTriple(T1 first, T2 second, T3 third) {
        this.first = first;
        this.second = second;
        this.third = third;
        this.hash = makeHash();
    }

    @Override
    public boolean equals(Object obj) {
        if (obj == null)
            return false;
        if (!(obj.getClass().equals(this.getClass())))
            return false;
        AdaptableTriple<?, ?, ?> other = (AdaptableTriple<?, ?, ?>) obj;
        if (other.first != first && (other.first == null || !other.first.equals(first)))
            return false;
        if (other.second != second && (other.second == null || !other.second.equals(second)))
            return false;
        if (other.third != third && (other.third == null || !other.third.equals(third)))
            return false;
        return true;
    }

    @Override
    public int hashCode() {
        return hash;
    }

    @Override
    public String toString() {
        return "<"+first+", "+second+", "+third+">";
    }

    private int makeHash() {
        return (first == null ? 0 : first.hashCode())
                + (second == null ? 0 : second.hashCode())*31
                + (third == null ? 0 : third.hashCode())*31*31;
    }

    private Object adapt(Object o, Class<?> clazz) {
        if (clazz.isInstance(o))
            return o;
        if (o instanceof IAdaptable)
            return ((IAdaptable) o).getAdapter(clazz);
        return null;
    }

    @SuppressWarnings("rawtypes")
    @Override
    public Object getAdapter(Class adapter) {
        Object o = adapt(first, adapter);
        if (o == null)
            o = adapt(second, adapter);
        if (o == null)
            o = adapt(third, adapter);
        return o;
    }

}
