/*******************************************************************************
 * Copyright (c) 2007, 2017 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *     Semantum Oy - #7178 Use NameLabelUtil
 *******************************************************************************/
package org.simantics.diagram.query;

import java.util.ArrayList;
import java.util.Collection;

import org.simantics.NameLabelUtil;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.common.request.ResourceRead;
import org.simantics.db.common.utils.NameUtils;
import org.simantics.db.exception.DatabaseException;
import org.simantics.diagram.stubs.DiagramResource;
import org.simantics.layer0.Layer0;
import org.simantics.layer0.utils.binaryPredicates.OrderedSetElementsPredicate;
import org.simantics.modeling.ModelingResources;
import org.simantics.structural.stubs.StructuralResource2;
import org.simantics.structural2.utils.StructuralUtils;

/**
 * @author Hannu Niemist&ouml;
 * @author Tuukka Lehtonen
 */
public class FlagTextQuery extends ResourceRead<String[]> {

    private static final boolean DEBUG = false;

    static final String[] NO_RESULT = new String[] { "?", "?" };

    public FlagTextQuery(Resource flag) {
        super(flag);
    }

    @Override
    public String[] perform(ReadGraph g) throws DatabaseException {
        Layer0 l0 = Layer0.getInstance(g);
        StructuralResource2 sr = StructuralResource2.getInstance(g);
        DiagramResource dr = DiagramResource.getInstance(g);

        if (DEBUG)
            System.out.println(getClass().getSimpleName() + "(" + NameUtils.getSafeName(g, resource) + ")");

        Resource connectionJoin = g.getPossibleObject(resource, dr.FlagIsJoinedBy);
        if (DEBUG)
            System.out.println(getClass().getSimpleName() + " connectionJoin: " + NameUtils.getSafeName(g, connectionJoin));
        if (connectionJoin == null)
            return NO_RESULT;

        Resource diagram = getFirstOwnerDiagram(g, resource);
        if (diagram == null)
            return NO_RESULT;

        // Get structural composite of flag's diagram.
        String DiagramToCompositeURI = ModelingResources.URIs.DiagramToComposite;
        Resource DiagramToComposite = g.getResource(DiagramToCompositeURI);
        Resource composite = g.getPossibleObject(diagram, DiagramToComposite);
        if (DEBUG)
            System.out.println(getClass().getSimpleName() + " composite: " + NameUtils.getSafeName(g, composite));
        if (composite == null)
            return NO_RESULT;

        // Find the connection that is on this diagram's composite and exclude
        // it from browsing of connections related to the connection join.
        Collection<Resource> connectionsOnFlagDiagram = findConnectionOnCompositeFromJoin(g, composite, connectionJoin);

        for (Resource connection : StructuralUtils.getRelatedConnectionsOfConnectionJoin(g, connectionJoin, connectionsOnFlagDiagram)) {
            if (DEBUG)
                System.out.println(getClass().getSimpleName() + " connection: " + NameUtils.getSafeName(g, connection));
            for (Resource component : g.getObjects(connection, sr.Connects)) {
                if (DEBUG)
                    System.out.println(getClass().getSimpleName() + " connects: " + NameUtils.getSafeName(g, component));
                // TODO how to correctly resolve flag text in cases where
                // the other diagram's connection has many attachments ?
                return new String[] {
                        getSafeLabel(g, g.getSingleObject(component, l0.PartOf)),
                        getSafeLabel(g, component)
                };
            }
        }

        return NO_RESULT;
    }

    private Resource getFirstOwnerDiagram(ReadGraph graph, Resource diagramPart) throws DatabaseException {
        Resource diagram = null;
        for (Resource temp : OrderedSetElementsPredicate.INSTANCE.getSubjects(graph, resource)) {
            diagram = temp;
            break;
        }
        if (DEBUG)
            System.out.println(getClass().getSimpleName() + " getFirstOwnerDiagram(" + NameUtils.getSafeName(graph, diagramPart) + "): " + NameUtils.getSafeName(graph, diagram));
        return diagram;
    }

    /**
     * Find the connections that are related to the specified connection join
     * and on the specified composite.
     * 
     * @param graph
     * @param composite
     * @param connectionJoin
     * @return
     * @throws DatabaseException
     */
    private Collection<Resource> findConnectionOnCompositeFromJoin(ReadGraph graph, Resource composite, Resource connectionJoin) throws DatabaseException {
        StructuralResource2 sr = StructuralResource2.getInstance(graph);
        Collection<Resource> result = new ArrayList<Resource>(1);
        for (Resource connection : graph.getObjects(connectionJoin, sr.Joins)) {
            if (DEBUG)
                System.out.println(getClass().getSimpleName() + " joins connection: " + NameUtils.getSafeName(graph, connection));
            if (StructuralUtils.isConnectionInComposite(graph, connection, composite))
                result.add(connection);
        }
        return result;
    }

    public static String getSafeLabel(ReadGraph graph, Resource r) throws DatabaseException {
        return NameLabelUtil.modalName(graph, r);
    }

}
