package org.simantics.diagram.flag;

import org.simantics.Simantics;
import org.simantics.db.ReadGraph;
import org.simantics.db.RequestProcessor;
import org.simantics.db.Resource;
import org.simantics.db.Session;
import org.simantics.db.WriteGraph;
import org.simantics.db.common.primitiverequest.PossibleObject;
import org.simantics.db.common.request.ResourceRead;
import org.simantics.db.common.request.WriteRequest;
import org.simantics.db.exception.DatabaseException;
import org.simantics.diagram.stubs.DiagramResource;
import org.simantics.utils.ui.ErrorLogger;

/**
 * @author Tuukka Lehtonen
 */
public class DiagramFlagPreferences {

    /**
     * Return the resource describing the currently active flag labeling scheme
     * from the specified resource. The default intended resource for checking
     * the scheme from is the active project.
     */
    public static class ReadFlagLabelingSchemeResource extends ResourceRead<Resource> {

        public ReadFlagLabelingSchemeResource(Resource resource) {
            super(resource);
        }

        @Override
        public Resource perform(ReadGraph graph) throws DatabaseException {
            DiagramResource DIA = DiagramResource.getInstance(graph);
            Resource scheme = graph.syncRequest(new PossibleObject(resource,
                    DIA.UsesLocalFlagLabelingScheme));
            return scheme;
        }

    }

    /**
     * Adapt {@link FlagLabelingScheme} instance from the resource describing
     * the currently active flag labeling scheme according to
     * {@link ReadFlagLabelingSchemeResource}.
     */
    public static class ReadFlagLabelingScheme extends ResourceRead<FlagLabelingScheme> {

        public ReadFlagLabelingScheme(Resource resource) {
            super(resource);
        }

        @Override
        public FlagLabelingScheme perform(ReadGraph graph) throws DatabaseException {
            DiagramResource DIA = DiagramResource.getInstance(graph);
            Resource scheme = graph.syncRequest(new ReadFlagLabelingSchemeResource(resource));
            if (scheme == null)
                return graph.adapt(DIA.FlagLabelingScheme_Alphabetical, FlagLabelingScheme.class);
            return graph.adapt(scheme, FlagLabelingScheme.class);
        }

    }

    public static Resource getActiveFlagLabelingSchemeResource(RequestProcessor processor) throws DatabaseException {
        Resource project = Simantics.getProjectResource();
        if (project == null)
            throw new IllegalStateException("no project is active");

        return processor.syncRequest(new ReadFlagLabelingSchemeResource(project));
    }

    public static FlagLabelingScheme getActiveFlagLabelingScheme(RequestProcessor processor) throws DatabaseException {
        Resource project = Simantics.getProjectResource();
        if (project == null)
            throw new IllegalStateException("no project is active");

        return processor.syncRequest(new ReadFlagLabelingScheme(project));
    }

    public static void setProjectFlagLabelingScheme(Resource scheme) {
        Resource project = Simantics.getProjectResource();
        if (project == null)
            throw new IllegalStateException("no project is active");
        setFlagLabelingScheme(Simantics.getSession(), project, scheme);
    }

    public static void setFlagLabelingScheme(Session session, final Resource forTarget, final Resource scheme) {
        session.asyncRequest(new WriteRequest() {
            @Override
            public void perform(WriteGraph graph) throws DatabaseException {
                setFlagLabelingScheme(graph, forTarget, scheme);
            }
        }, parameter -> {
            if (parameter != null)
                ErrorLogger.defaultLogError(parameter);
        });
    }

    public static void setFlagLabelingScheme(WriteGraph graph, Resource forTarget, Resource scheme) throws DatabaseException {
        DiagramResource DIA = DiagramResource.getInstance(graph);
        graph.deny(forTarget, DIA.UsesLocalFlagLabelingScheme);
        graph.claim(forTarget, DIA.UsesLocalFlagLabelingScheme, scheme);
    }

}
