/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.diagram.flag;

import java.awt.Shape;
import java.awt.geom.Path2D;
import java.awt.geom.Rectangle2D;

import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.exception.DatabaseException;
import org.simantics.diagram.query.DiagramRequests;
import org.simantics.g2d.elementclass.FlagClass;
import org.simantics.g2d.elementclass.FlagClass.Mode;
import org.simantics.g2d.elementclass.FlagClass.Type;
import org.simantics.g2d.utils.Alignment;
import org.simantics.structural2.modelingRules.IModelingRules;

/**
 * @author Tuukka Lehtonen
 */
public class BasicFlagType extends AbstractFlagType {

    public BasicFlagType(Resource flag, IModelingRules modelingRules) {
        super(flag, modelingRules);
    }

    @Override
    public FlagInfo getInfo(ReadGraph graph) throws DatabaseException {
        Type type = getType(graph);
        Mode mode = getMode(graph);

        return FlagInfoBuilder.fill(type)
        .shape(getShape(graph, type, mode))
        .text(getText(graph))
        .textArea(getArea(graph, type, mode))
        .horizontalAlignment(Alignment.LEADING)
        .verticalAlignment(Alignment.CENTER)
        .create();
    }

    protected Rectangle2D getArea(ReadGraph graph, Type type, Mode mode) {
        return getArea(type, mode);
    }

    public static Rectangle2D getArea(Type type, Mode mode) {
        final double width = FlagClass.DEFAULT_WIDTH;
        final double height = FlagClass.DEFAULT_HEIGHT;
        final double beakLength = FlagClass.getBeakLength(height, FlagClass.DEFAULT_BEAK_ANGLE);
        return getArea(type, mode, width, height, beakLength);
    }

    public static Rectangle2D getArea(Type type, Mode mode, double width, double height, double beakLength) {
        double x = 0;
        double y = -height / 2;
        double w = width;
        double h = height;

        if (type == Type.In) {
            if (mode instanceof FlagClass.External) {
                x = -width-beakLength;
            } else if (mode instanceof FlagClass.Internal) {
                x = -beakLength;
                w = beakLength;
            }
        } else if (type == Type.Out) {
            if (mode instanceof FlagClass.Internal) {
                w = beakLength;
            }
        }

        return new Rectangle2D.Double(x, y, w, h);
    }

    protected Shape getShape(ReadGraph graph, Type type, Mode mode) throws DatabaseException {
        return getShape(type, mode);
    }

    public static Shape getShape(Type type, Mode mode) {
        final double width = FlagClass.DEFAULT_WIDTH;
        final double height = FlagClass.DEFAULT_HEIGHT;
        final double beakLength = FlagClass.getBeakLength(height, FlagClass.DEFAULT_BEAK_ANGLE);
        Path2D path = new Path2D.Double();
        FlagClass.createFlagShape(path, type, mode, width, height, beakLength);
        return path;
    }

    protected String[] getText(ReadGraph graph) throws DatabaseException {
        return graph.syncRequest(DiagramRequests.getFlagText(flag));
    }

}