package org.simantics.diagram.copypaste;

import java.awt.geom.Point2D;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import org.simantics.Simantics;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.common.ResourceArray;
import org.simantics.db.common.request.PossibleIndexRoot;
import org.simantics.db.common.request.ResourceRead;
import org.simantics.db.common.request.UniqueRead;
import org.simantics.db.common.utils.NameUtils;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.exception.ValidationException;
import org.simantics.db.management.ISessionContext;
import org.simantics.diagram.elements.DiagramNodeUtil;
import org.simantics.diagram.handler.CopyPasteHandler;
import org.simantics.diagram.query.DiagramRequests;
import org.simantics.g2d.canvas.impl.CanvasContext;
import org.simantics.g2d.diagram.DiagramHints;
import org.simantics.g2d.diagram.participant.Selection;
import org.simantics.g2d.element.IElement;
import org.simantics.g2d.page.DiagramDesc;
import org.simantics.g2d.participant.MouseUtil;
import org.simantics.g2d.scenegraph.ICanvasSceneGraphProvider;
import org.simantics.layer0.Layer0;
import org.simantics.layer0.utils.triggers.IActivation;
import org.simantics.layer0.utils.triggers.IActivationManager;
import org.simantics.modeling.ModelingResources;
import org.simantics.scenegraph.g2d.events.MouseEvent.MouseMovedEvent;
import org.simantics.scenegraph.g2d.events.command.Command;
import org.simantics.scenegraph.g2d.events.command.Commands;
import org.simantics.scenegraph.g2d.snap.GridSnapAdvisor;
import org.simantics.structural2.StructuralVariables;
import org.simantics.utils.datastructures.Pair;
import org.simantics.utils.datastructures.hints.IHintContext;
import org.simantics.utils.threads.ThreadUtils;
import org.simantics.utils.threads.WorkerThread;
import org.slf4j.LoggerFactory;

public class ElementCopyPaster {
    private static final org.slf4j.Logger LOGGER = LoggerFactory.getLogger(ElementCopyPaster.class);

    private final WorkerThread thread = new WorkerThread("Copy Paster Diagram Thread");
    private final CanvasContext ctx = new CanvasContext(thread);
    private IActivation activation;

    private Resource diagram;
    private ICanvasSceneGraphProvider provider;

    public ElementCopyPaster(Resource diagram) throws Exception {
        this.diagram = diagram;
        thread.start();

        final ISessionContext sessionContext = Simantics.getSessionContext();

        DiagramDesc diagramDesc = sessionContext.getSession().syncRequest(DiagramRequests.getDiagramDesc(diagram));

        if (diagramDesc != null) {
            IHintContext h = ctx.getDefaultHintContext();
            h.setHint(DiagramHints.SNAP_ADVISOR, new GridSnapAdvisor(diagramDesc.getGridSize()));
        }

        // IMPORTANT: Load diagram in a different thread than the canvas context thread!

        Runnable r = new Runnable() {
            @Override
            public void run() {
                try {
                    Pair<Resource, String> modelAndRVI = sessionContext.getSession()
                            .syncRequest(new UniqueRead<Pair<Resource, String>>() {
                                @Override
                                public Pair<Resource, String> perform(ReadGraph graph) throws DatabaseException {
                                    Resource model = resolveModel(graph, diagram);
                                    return new Pair<Resource, String>(model, resolveRVI(graph, model, diagram));
                                }
                            });
                    provider = DiagramNodeUtil.loadSceneGraphProviderForDiagram(ctx,
                            modelAndRVI.first, diagram, modelAndRVI.second/*, __context*/);
                    
                    IActivationManager activationManager = sessionContext.getSession().peekService(IActivationManager.class);
                    if (activationManager != null) {
                        activation = activationManager.activate(diagram);
                    }

                } catch (DatabaseException | InterruptedException e) {
                    LOGGER.error("Failed to load diagram.", e);
                }
            }
        };

        Thread t = new Thread(r);
        t.start();
        t.join();
    }

    private void copyOrCutElementsInternal(List<Resource> elements, boolean cut) {
        ThreadUtils.asyncExec(thread, new Runnable() {
            @Override
            public void run() {
                CopyPasteHandler cph = ctx.getAtMostOneItemOfClass(CopyPasteHandler.class);
                
        
                try {
                    Collection<IElement> ielements = Simantics.sync(new UniqueRead<Collection<IElement>>() {
        
                        @Override
                        public Collection<IElement> perform(ReadGraph graph) throws DatabaseException {
                            Collection<IElement> result = new ArrayList<>();
                            for (Resource er : elements) {
        
                                IElement e = DiagramNodeUtil.findElement(ctx, er);
                                result.add(e);
                            }
                            return result;
        
                        }
                    });

                    Selection selection = ctx.getAtMostOneItemOfClass(Selection.class);
                    selection.setSelection(0, ielements);

                    cph.initiateCopy(cut);

                } catch (DatabaseException e) {
                    LOGGER.error("Failed to copy/cut elements.", e);
                }
            }
        });

    }
    
    private void pasteElementsInternal(double x, double y, Command command) {
        ThreadUtils.syncExec(thread, new Runnable() {
            @Override
            public void run() {
                CopyPasteHandler cph = ctx.getAtMostOneItemOfClass(CopyPasteHandler.class);
                MouseUtil mouseUtil = ctx.getSingleItem(MouseUtil.class);
                mouseUtil.handleMouseEvent(new MouseMovedEvent(null, 0, 0, 0, 0, new Point2D.Double(x, y), null));

                cph.initiatePaste(command);

            }
        });
        
    }

    private static Resource resolveModel(ReadGraph graph, Resource diagram) throws DatabaseException {
        ModelingResources mod = ModelingResources.getInstance(graph);
        Resource composite = graph.getSingleObject(diagram, mod.DiagramToComposite);
        Resource model = graph.syncRequest(new PossibleIndexRoot(composite));
        if (model == null)
            throw new ValidationException("no model found for composite " + NameUtils.getSafeName(graph, composite));
        return model;
    }

    private static String resolveRVI(ReadGraph graph, Resource model, Resource diagram) throws DatabaseException {
        /*
         * ModelingResources mod = ModelingResources.getInstance(graph); Resource
         * composite = graph.getSingleObject(diagram, mod.DiagramToComposite); String
         * modelURI = graph.getURI(model); String compositeURI =
         * graph.getURI(composite); return compositeURI.substring(modelURI.length());
         */

        ModelingResources mod = ModelingResources.getInstance(graph);
        Resource composite = graph.getSingleObject(diagram, mod.DiagramToComposite);
        final ResourceArray compositePath = StructuralVariables.getCompositeArray(graph, composite);
        final ResourceArray variablePath = compositePath.removeFromBeginning(1);
        return StructuralVariables.getRVI(graph, variablePath);
    }

    public Resource getDiagram() {
        return diagram;
    }

    private void dispose() {
        if (activation != null)
            activation.deactivate();
        if (provider != null)
            provider.dispose();
        if (thread != null)
            thread.stopDispatchingEvents(true);
        if (ctx != null)
            ctx.dispose();
    }

    private static ElementCopyPaster source;

    public synchronized static void copyElements(List<Resource> elements) throws Exception {
        copyOrCutElements(elements, false);
    }

    public synchronized static void cutElements(List<Resource> elements) throws Exception {
        copyOrCutElements(elements, true);
    }

    private static void copyOrCutElements(List<Resource> elements, boolean cut) throws Exception {
        if (elements.isEmpty()) {
            if (source != null) {
                source.dispose();
                source = null;
            }
        } else {
            Resource first = elements.get(0);
            Resource diagram = Simantics.sync(new ResourceRead<Resource>(first) {
                @Override
                public Resource perform(ReadGraph graph) throws DatabaseException {
                    Layer0 L0 = Layer0.getInstance(graph);
                    return graph.getSingleObject(first, L0.PartOf);
                }
            });

            if (source == null || source.getDiagram() != diagram) {
                if (source != null) {
                     source.dispose();
                }
                source = new ElementCopyPaster(diagram);
            }
            source.copyOrCutElementsInternal(elements, cut);
        }
    }

    public synchronized static void pasteElements(Resource targetDiagram, double x, double y) throws Exception {
       pasteElementsWithCommand(targetDiagram, x, y, Commands.PASTE);
    }

    public synchronized static void pasteElementsWithCommand(Resource targetDiagram, double x, double y, Command command) throws Exception {
        if (source != null && source.getDiagram() != targetDiagram) {
            ElementCopyPaster target = new ElementCopyPaster(targetDiagram);
            try {
                target.pasteElementsInternal(x, y, command);
            } finally {
                target.dispose();
            }
        } else {
            source.pasteElementsInternal(x, y, command);
        }
    }

}
