/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.diagram.contribution;

import org.eclipse.core.runtime.Platform;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.action.ActionContributionItem;
import org.eclipse.jface.action.IContributionItem;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.ui.IWorkbenchPart;
import org.simantics.db.ReadGraph;
import org.simantics.db.Session;
import org.simantics.db.exception.DatabaseException;
import org.simantics.g2d.canvas.ICanvasContext;
import org.simantics.g2d.diagram.participant.Selection;
import org.simantics.ui.contribution.DynamicMenuContribution;
import org.simantics.utils.threads.ThreadUtils;
import org.simantics.utils.ui.BundleUtils;
import org.simantics.utils.ui.workbench.WorkbenchUtils;


/**
 * @author Antti Villberg
 */
abstract public class DiagramDynamicMenuContribution<T> extends DynamicMenuContribution {

	private static final IContributionItem[] NONE = {};

	private ICanvasContext canvas;

	@Override
	public void fill(Menu menu, int index) {
		// Need to grab active part here, we're still in the SWT thread.
		IWorkbenchPart activePart = WorkbenchUtils.getActiveWorkbenchPart();
		if (activePart == null)
			return;
		ICanvasContext ctx = (ICanvasContext) activePart.getAdapter(ICanvasContext.class);
		if (ctx == null)
			return;

		this.canvas = ctx;
		try {
			super.fill(menu, index);
		} finally {
			this.canvas = null;
		}
	}

	@Override
	protected IContributionItem[] getContributionItems(ReadGraph graph, Object[] selection) throws DatabaseException {

		T input = computeInput(graph, selection);

		if(input == null) return NONE;

		return new IContributionItem[] {
				new ActionContributionItem(new Helper(graph.getSession(), canvas, input))
		};

	}

	public class Helper extends Action {

		private final T input;
		protected final Session        session;
		protected final ICanvasContext context;

		public Helper(Session session, ICanvasContext context, T input) {
			super(getName(), getImage());
			this.session = session;
			this.context = context;
			this.input = input;
		}

		@Override
		public void run() {
			perform(session, context, input);
			ThreadUtils.asyncExec(context.getThreadAccess(), new Runnable() {
				@Override
				public void run() {
					if (context.isDisposed())
						return;
					Selection selection = context.getAtMostOneItemOfClass(Selection.class);
					if (selection != null) {
						// This prevents workbench selection from being left over.
						// Also prevents scene graph crap from being left on the screen.
						selection.clear(0);
					}
				}
			});
		}

	}

	protected ImageDescriptor silk(String name) {
		return BundleUtils.getImageDescriptorFromBundle(Platform.getBundle("com.famfamfam.silk"), "/icons/" + name);
	}

	abstract protected T computeInput(ReadGraph graph, Object[] selection) throws DatabaseException;
	abstract protected void perform(Session session, ICanvasContext context, T input);
	abstract protected String getName();
	abstract protected ImageDescriptor getImage();

}
