/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.diagram.content;

import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.simantics.db.Resource;
import org.simantics.utils.datastructures.MapList;

/**
 * @author Tuukka Lehtonen
 */
public class DiagramContents {

    /**
     * Z-ordered list of ALL direct diagram elements.
     */
    public List<Resource>            elements                = Collections.emptyList();

    /**
     * The set of normal elements in the diagram contents list.
     */
    public Set<Resource>             nodeSet                 = Collections.emptySet();

    public Set<Resource>             connectionSet           = Collections.emptySet();
    public Set<EdgeResource>         connectionSegments      = Collections.emptySet();
    public Set<Resource>             branchPoints            = Collections.emptySet();

    public Set<Resource>             routeGraphConnectionSet = Collections.emptySet();
    public Set<EdgeResource>         routeLinks              = Collections.emptySet();
    public Set<Resource>             routeLines              = Collections.emptySet();
    public Set<Resource>             routePoints             = Collections.emptySet();

    public Map<Object, Resource>     partToConnection        = Collections.emptyMap();
    public MapList<Resource, Object> connectionToParts       = new MapList<Resource, Object>();

    private <T> void calculateChanges(Set<T> from, Set<T> to, Map<T, Change> result) {
        for (T e : to)
            if (!from.contains(e))
                result.put(e, Change.ADDED);
//            else
//                result.put(e, Change.POSSIBLY_MODIFIED);
        for (T e : from)
            if (!to.contains(e))
                result.put(e, Change.REMOVED);
    }

    private <T> void addAll(Set<T> to, Map<T, Change> result) {
        for (T e : to)
            result.put(e, Change.ADDED);
    }

    public DiagramContentChanges differenceFrom(DiagramContents from) {
        DiagramContentChanges result = new DiagramContentChanges();
        if (this == from)
            return result;
        if (from == null) {
            // Both nodes and connections are considered elements of the diagram.
            addAll(nodeSet, result.elements);
            addAll(connectionSet, result.elements);
            addAll(routeGraphConnectionSet, result.elements);
            addAll(nodeSet, result.nodes);
            addAll(connectionSet, result.connections);
            addAll(connectionSegments, result.connectionSegments);
            addAll(branchPoints, result.branchPoints);
            addAll(routeGraphConnectionSet, result.routeGraphConnections);
            addAll(routeLinks, result.routeLinks);
            addAll(routeLines, result.routeLines);
            addAll(routePoints, result.routePoints);
        } else {
            // Both nodes and connections are considered elements of the diagram.
            calculateChanges(from.nodeSet, nodeSet, result.elements);
            calculateChanges(from.connectionSet, connectionSet, result.elements);
            calculateChanges(from.routeGraphConnectionSet, routeGraphConnectionSet, result.elements);
            calculateChanges(from.nodeSet, nodeSet, result.nodes);
            calculateChanges(from.connectionSet, connectionSet, result.connections);
            calculateChanges(from.connectionSegments, connectionSegments, result.connectionSegments);
            calculateChanges(from.branchPoints, branchPoints, result.branchPoints);
            calculateChanges(from.routeGraphConnectionSet, routeGraphConnectionSet, result.routeGraphConnections);
            calculateChanges(from.routeLinks, routeLinks, result.routeLinks);
            calculateChanges(from.routeLines, routeLines, result.routeLines);
            calculateChanges(from.routePoints, routePoints, result.routePoints);

            if (!from.elements.equals(elements))
                result.markElementOrderChanged();
        }
        return result;
    }

    @Override
    public int hashCode() {
        final int prime = 31;
        int result = 1;
        result = prime * result + ((routeGraphConnectionSet == null) ? 0 : routeGraphConnectionSet.hashCode());
        result = prime * result + ((connectionSet == null) ? 0 : connectionSet.hashCode());
        result = prime * result + ((elements == null) ? 0 : elements.hashCode());
        return result;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj)
            return true;
        if (obj == null)
            return false;
        if (getClass() != obj.getClass())
            return false;
        DiagramContents other = (DiagramContents) obj;
        if (!elements.equals(other.elements))
            return false;
        if (!connectionSegments.equals(other.connectionSegments))
            return false;
        if (!branchPoints.equals(other.branchPoints))
            return false;
        if (!routeLinks.equals(other.routeLinks))
            return false;
        if (!routeLines.equals(other.routeLines))
            return false;
        if (!routePoints.equals(other.routePoints))
            return false;
        return true;
    }

    @Override
    public String toString() {
        return getClass().getSimpleName() + "[elements=" + elements.size()
        + ", nodes=" + nodeSet.size()
        + ", connections=" + connectionSet.size()
        + ", branch points=" + branchPoints.size()
        + ", connection segments=" + connectionSegments.size()
        + ", routegraph connections=" + routeGraphConnectionSet.size()
        + ", routegraph links=" + routeLinks.size()
        + ", route lines=" + routeLines.size()
        + ", route points=" + routePoints.size()
        + "]";
    }

}